package main

import (
	"errors"
	"fmt"
	"log"
	"net/url"
	"strings"

	"code.justin.tv/video/gotranscoder/pkg/adinsertion"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
)

// receiveAdBreakMessages creates a client listening for
// AdBreakRequests from pubsub and runs the client, pushing messages
// into the adBreakPluginsContainer.
func receiveAdBreakMessages() error {
	settings := &adinsertion.Settings{
		URL:       cfg.Ads.PubsubURL,
		Proxy:     cfg.Ads.Proxy,
		Topic:     cfg.Ads.Topic,
		AuthToken: "",
	}

	if channel == nil {
		return errors.New("channel is nil, so unable to connect to a pubsub topic")
	}

	// TODO: this hack needs to be replaced by a more generic solution.
	//		We are only adding this in order to support ads for low-latency dark launch
	// Anything channel that does not indicate low-latency darklaunch should continue to
	// work as it did before, i.e. ch = *channel, err = nil
	ch, err := getChannelName(*channel, *rtmpUrl)
	if err != nil {
		return fmt.Errorf("Could not resolve channel name for '%s': %v", ch, err)
	}

	client, err := adinsertion.NewAdInsertionClient(settings, ch)
	if err != nil {
		return fmt.Errorf("unable to make ad insertion client: %s", err)
	}
	log.Printf("[AD INSERTION] client created on topic %s", fmt.Sprintf(settings.Topic, ch))

	onExit.Add(func() {
		err := client.Close()
		if err != nil {
			log.Printf("[AD INSERTION] error while closing ad insertion client: %s", err)
		}
	})

	go func() {
		for {
			req, err := client.Next()
			if err != nil {
				log.Printf("[AD INSERTION] error receiving ad insertion message: %s", err)
				continue
			}
			if req == nil {
				// req == nil and err == nil means we've close the client
				return
			}
			log.Printf("[AD INSERTION] message received")
			statsd.Inc(constAdBreakRequest, 1, 1)

			var val interface{} = req
			adBreakPluginsContainer.Eval(&val)
		}
	}()
	return nil
}

func getChannelName(ch, rtmpUrl string) (string, error) {
	// If this is a darklaunch channel it must begin with this prefix
	if strings.HasPrefix(ch, lowLatencyDarkLaunchPrefix) {
		u, err := url.Parse(rtmpUrl)
		if err != nil {
			return "", err
		}
		ch := strings.TrimPrefix(u.Path, "/app/live_user_")
		if ch == u.Path {
			return "", fmt.Errorf("Failed to extract a valid channel from the RTMP URL - '%s'", rtmpUrl)
		}
		return ch, nil
	}

	// non-LVS channels should continue to function as normal
	return ch, nil
}
