package main

import (
	"fmt"
	"os"
	"reflect"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/twitchtranscoder"
	"code.justin.tv/video/gotranscoder/pkg/vod"
)

// InitElasticSearch holds the settings for elastic search
func initElasticSearch() {
	es = elasticsearch.Settings{
		Host:               cfg.ElasticSearch.Host,
		Port:               cfg.ElasticSearch.Port,
		SampleRate:         cfg.ElasticSearch.SampleRate,
		IndexRetentionDays: cfg.ElasticSearch.IndexRetentionDays,
	}

	es.Initialize()
}

// ParseDebugLine extracts debug information for libavtwitch's output
func ParseDebugLine(line string) {
	debugData := twitchtranscoder.ParseDebug(line)

	if debugData != nil && cfg.ElasticSearch.Enable {
		msg := elasticsearch.Log{
			MessageType: "Debug",
			Channel:     *channel,
			ChannelID:   *channelID,
			Data:        debugData,
		}

		es.WriteLog(&msg, false)
		statsd.Inc(constEsUnsampledLog, 1, 1)
	}
}

// ParseJSONLine will try to decode json from an output line coming form libavtwitch
// This is where it'll be decided how to handle the output of each type
func ParseJSONLine(data []byte) {
	line, _ := twitchtranscoder.ParseLine(data)

	if line != nil && cfg.ElasticSearch.Enable {
		logEntry := elasticsearch.Log{
			MessageType: fmt.Sprintf("%s", reflect.TypeOf(line)),
			Channel:     *channel,
			ChannelID:   *channelID,
			Data:        line,
		}

		switch line.(type) {
		case avdata.Segment:
			segmentPluginsContainer.Eval(&line)
			es.WriteLog(&logEntry, true)
			statsd.Inc(constEsSampledLog, 1, 1)
		case avdata.Codec, avdata.AvVersions:
			codecPluginsContainer.Eval(&line)
			es.WriteLog(&logEntry, false)
			statsd.Inc(constEsUnsampledLog, 1, 1)
		case avdata.Thumbnail:
			es.WriteLog(&logEntry, true)
			statsd.Inc(constEsSampledLog, 1, 1)
			thumbPluginsContainer.Eval(&line)
		case avdata.Sprite:
			es.WriteLog(&logEntry, true)
			statsd.Inc(constEsSampledLog, 1, 1)
			thumbPluginsContainer.Eval(&line)
		case avdata.TwitchTranscoderLog:
			es.WriteLog(&logEntry, false)
			transcoderLog, ok := line.(avdata.TwitchTranscoderLog)
			if ok {
				statsd.Inc(fmt.Sprintf("%s.%s.%s", twitchtranscoder.TranscoderlogPrefix, transcoderLog.TTLData.Type, transcoderLog.TTLData.Topic), 1, 1)
			}
		case avdata.StreamLog:
			es.WriteLog(&logEntry, true)
			streamlogPluginsContainer.Eval(&line)
			statsd.Inc(constEsSampledLog, 1, 1)
		case avdata.Fmp4InitSegment:
			segmentPluginsContainer.Eval(&line)
			es.WriteLog(&logEntry, true)
			statsd.Inc(constEsSampledLog, 1, 1)
		default:
			es.WriteLog(&logEntry, true)
			statsd.Inc(constEsSampledLog, 1, 1)
		}
	}
}

// LogHostConfiguration the current host settings
func LogHostConfiguration() {
	hostName, _ := os.Hostname()

	data := map[string]interface{}{
		"Hostname":               hostName,
		"TwitchTranscoder":       transcoder.Path,
		"sessionID":              sessionID,
		"listenerPort":           transcoder.ListenerPort,
		"transcodeID":            transcodeID,
		"channelPropertyID":      channelPropertyID,
		"broadcasTag":            broadcastTag,
		"rtmpURL":                encoderConfig.RtmpURL,
		"twitchTranscoderConfig": cfg.TwitchTranscoder,
		"statelessMode":          *statelessMode,
		"vodPusherEnabled":       *vodPusherEnabled,
		"nativeHeight":           *nativeHeight,
		"targetSegmentDuration":  cfg.Segment.TargetDuration,
		"encoderConfig":          fmt.Sprintf("%+v", encoderConfig),
		"transcodeProfile":       fmt.Sprintf("%+v", *transcodeProfile),
		"rtmpEndpoint":           fmt.Sprintf("%+v", *rtmpEndpoint),
	}

	msg := elasticsearch.Log{MessageType: "HostConfiguration",
		Channel:   *channel,
		ChannelID: *channelID,
		Data:      data,
	}

	es.WriteLog(&msg, false)
	statsd.Inc(constEsUnsampledLog, 1, 1)

}

// LogVodConfiguration to help track vod writes
func LogVodConfiguration(vodCfg *vod.PusherSettings) {
	data := map[string]interface{}{
		"BroadcastID":      vodCfg.BroadcastID,
		"VodKey":           vodCfg.VodKey,
		"QueueSize":        vodCfg.QueueSize,
		"NumThumbnails":    vodCfg.NumThumbnails,
		"ChunkFactor":      vodCfg.ChunkFactor,
		"PlaylistFileName": vodCfg.PlaylistFileName,
		"VinylHost":        vodCfg.VinylHost,
		"S3Bucket":         vodCfg.Bucket,
	}

	msg := elasticsearch.Log{MessageType: "VodConfiguration",
		Channel:   *channel,
		ChannelID: *channelID,
		Data:      data,
	}

	es.WriteLog(&msg, false)
	statsd.Inc(constEsUnsampledLog, 1, 1)

}

// LogVodPusherHealth sends pusher Health to elasticsearch
func LogVodPusherHealth(pusherHealth bool) {
	data := map[string]interface{}{
		"Healthy": pusherHealth,
	}

	msg := elasticsearch.Log{MessageType: "VodPusherHealth",
		Channel:   *channel,
		ChannelID: *channelID,
		Data:      data,
	}

	es.WriteLog(&msg, false)
	statsd.Inc(constEsUnsampledLog, 1, 1)
}
