package main

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/url"

	"code.justin.tv/video/gotranscoder/pkg/config"
	"code.justin.tv/video/gotranscoder/pkg/twitchtranscoder"
)

// Load the Transcode profiles and transcode quality definitions
func Load(transcodeProfiles string, transcodeQualities string) (*TranscodeProfiles, *TranscodeQualities, error) {

	var qualities = &TranscodeQualities{}
	var profiles = &TranscodeProfiles{}

	err := config.Load(transcodeProfiles, profiles)
	if err != nil {
		log.Println("Error loading Transcode profiles", err)
		return nil, nil, err
	}

	err = config.Load(transcodeQualities, qualities)
	if err != nil {
		log.Println("Error loading Transcode qualities", err)
		return nil, nil, err
	}
	return profiles, qualities, nil
}

// CullRedundantTranscodes removes transcodes which would be upscaling the origin video.
// Only leave transcodes which height is less than source
func CullRedundantTranscodes(encoder *EncoderConfig, videoHeight int32) {
	// In case goingest failed to populate the video height - don't cull transcodes
	if videoHeight == 0 {
		return
	}

	encoder.Transcodes = CullTranscodesByHeight(encoder.Transcodes, videoHeight)
}

// CullTranscodesByHeight returns a subset of transcodes that are <= height
func CullTranscodesByHeight(transcodes []TranscodeQualitySettings, height int32) []TranscodeQualitySettings {
	var culled []TranscodeQualitySettings

	for i := range transcodes {
		if transcodes[i].Height <= height {
			culled = append(culled, transcodes[i])
			log.Printf("Appending transcode quality: %s", transcodes[i].Label)
		}
	}
	return culled
}

// CullQuickSyncRenditions currently just calls CullQuickSyncRenditionsBasedOnFps
func CullQuickSyncRenditions(encoder *EncoderConfig, frameRate int) {
	encoder.Transcodes = CullQuickSyncRenditionsBasedOnFps(encoder.Transcodes, frameRate)
}

// CullQuickSyncRenditionsBasedOnFps removes QuickSync renditions based on framerate
// as we do not upscale. We make sure we always remove one of the 720p renditions
// based on the probe framerate. The default framerate we assume will be 60fps
// if the probe fails
func CullQuickSyncRenditionsBasedOnFps(transcodes []TranscodeQualitySettings, frameRate int) []TranscodeQualitySettings {
	//We need to remove one of the renditions for 720p videos based on the source fps
	var closestFrameRate = GetClosestFrameRate(frameRate)
	var culledTranscodes []TranscodeQualitySettings

	for i := range transcodes {
		if transcodes[i].Height == 720 {
			if transcodes[i].MaxFps == int32(closestFrameRate) {
				culledTranscodes = append(culledTranscodes, transcodes[i])
				log.Printf("Appending transcode quality: %s", transcodes[i].Label)
			}
		} else {
			culledTranscodes = append(culledTranscodes, transcodes[i])
			log.Printf("Appending transcode quality: %s", transcodes[i].Label)
		}
	}

	return culledTranscodes
}

// CullRenditionsBasedOnFps removes renditions from the encoder configuration
// which are above frameRate, clamped to values from GetClosestFrameRate
func CullRenditionsBasedOnFps(encoder *EncoderConfig, frameRate int) {
	var closestFrameRate = GetClosestFrameRate(frameRate)

	var culledTranscodes []TranscodeQualitySettings

	for i := range encoder.Transcodes {
		if encoder.Transcodes[i].MaxFps <= int32(closestFrameRate) {
			culledTranscodes = append(culledTranscodes, encoder.Transcodes[i])
			log.Printf("Appending transcode quality: %s", encoder.Transcodes[i].Label)
		}
	}

	encoder.Transcodes = culledTranscodes
}

// GetClosestFrameRate clamps the input framerate to valid values
func GetClosestFrameRate(frameRate int) int {
	if frameRate == 0 {
		log.Println("Using default video fps for culling renditions based on framerate")
		frameRate = defaultVideoFps
	}
	frameRate = roundOffFps(frameRate)
	if frameRate <= 48 {
		return 30
	}
	return 60
}

// Drop720p60Rendition returns a boolean variable to figure out whether we drop the 720p60 renditions
// If the function returns 'true' , we drop 720p60 , but retain 720p30
// If the function returns 'false' , we drop 720p30 , but retain 720p60
func Drop720p60Rendition(frameRate int, bitrate int64) bool {
	if frameRate == 30 {
		return true
	} else if bitrate < const720p60MinBitrateThreshold {
		return true
	}
	return false
}

// WriteEncoderConfig takes an encoder object and writes to disk the config.
func WriteEncoderConfig(cfg *EncoderConfig, basePath string) (string, error) {
	path := fmt.Sprintf("%s/encoder.json", basePath)
	data, err := json.Marshal(cfg)
	if err != nil {
		log.Printf("Error marshalling encoder config %s", err)
		return path, err
	}

	err = ioutil.WriteFile(path, data, 0644)
	if err != nil {
		log.Printf("Error writing encoder config to disk %s - %s", path, err)
		return path, err
	}

	return path, nil
}

// WriteTenfootConfig writes a config with the legacy playlist name -
// used for backwards compatibility in a couple servers running a tenfoot test
func WriteTenfootConfig(basePath string) error {
	path := fmt.Sprintf("%s/tenfoot.json", basePath)
	config := []byte(`{"playlist": "py-index-live.m3u8","version": "legacy"}"`)

	err := ioutil.WriteFile(path, config, 0644)
	if err != nil {
		log.Printf("Error writing tenfoot config to disk %s - %s", basePath, err)
		return err
	}

	return nil
}

//ChooseTranscoderProfile will select a profile stack based on url or job parameters
func ChooseTranscoderProfile(cfg *config.Values, profiles *TranscodeProfiles) string {
	// pri 1 - Rtmp stream url parameter
	if len(*broadcastTag) > 0 {
		streamParams, _ := url.ParseQuery(*broadcastTag)
		tags := streamParams["broadcast_tag"]

		if len(tags) > 0 {
			_, ok := profiles.Profiles[tags[0]]
			if ok {
				log.Println("Found broadcaster_tag overrride and matching transode profile - overrriding with", tags[0])
				*transcodeProfile = tags[0]
				return tags[0]
			}
		}
	} // broadcast tag

	// pri 2 - command line argument- TwitchTranscoder default
	if len(*transcodeProfile) > 0 {
		return *transcodeProfile
	}

	// pri 3 - config's default
	return cfg.TwitchTranscoder.TranscodeProfile
}

// GetBroadcaster attempts to parse the broadcaster field from the rtmp query string
func GetBroadcaster() string {
	if len(*broadcastTag) > 0 {
		streamParams, _ := url.ParseQuery(*broadcastTag)
		tags := streamParams["broadcaster"]

		if len(tags) > 0 {
			return tags[0]
		}
	}
	return ""
}

// GenerateEncoderConfig for the encoder - contains the full transcode settings
func GenerateEncoderConfig(cfg *config.Values, basePath string) (*EncoderConfig, error) {
	var encoding EncoderConfig

	profiles, qualities, err := Load(*configProfiles, *configQualities)
	if err != nil {
		log.Println("Error loading Transcode settings", err)
		return nil, err
	}

	// Audio
	encoding.Audio.Label = constTranscodeAudioOnly
	encoding.Audio.Path = fmt.Sprintf(constTranscodePathAudio, basePath)
	encoding.Audio.OptimizeTs = constTranscodeOptimizeTs
	encoding.Audio.Prefix = cfg.TwitchTranscoder.ChunkPrefix
	encoding.Audio.HlsValidation = constTranscodeHlsValidation

	// Thumbs
	encoding.Thumbs.Label = constTranscodeThumb
	encoding.Thumbs.Path = fmt.Sprintf(constTranscodePathThumb, basePath)
	encoding.Thumbs.Delay = cfg.TwitchTranscoder.ThumbDelay
	encoding.Thumbs.Interval = cfg.TwitchTranscoder.ThumbInterval
	encoding.Thumbs.FileNamePrefix = constTranscodeThumb

	// Storyboards/Spritesheets generation
	if *vodPusherEnabled {
		spriteSettings := cfg.TwitchTranscoder.Sprites
		if spriteSettings.ThumbnailHeight < spriteMinThumbnailHeight {
			log.Printf("Invalid sprite ThumbnailHeight got %d needs to be above %d",
				spriteSettings.ThumbnailHeight, spriteMinThumbnailHeight)
		} else if spriteSettings.ThumbnailWidth < spriteMinThumbnailWidth {
			log.Printf("Invalid sprite ThumbnailWidth got %d needs to be above %d",
				spriteSettings.ThumbnailWidth, spriteMinThumbnailWidth)
		} else if spriteSettings.IntervalBetweenThumbnails < spriteMinIntervalMilliseconds {
			log.Printf("Invalid sprite IntervalBetweenThumbnails got %d needs to be above %d",
				spriteSettings.IntervalBetweenThumbnails, spriteMinIntervalMilliseconds)
		} else if spriteSettings.MaxSpriteRows < 1 || spriteSettings.MaxSpriteColumns < 1 {
			log.Printf("Invalid spritesheet rows x cols dimensions, given %dx%d, requires at least 1x1",
				spriteSettings.MaxSpriteRows, spriteSettings.MaxSpriteColumns)
		} else {
			encoding.Sprites = spriteSettings

			log.Printf("'label' and 'file_base' are configured by go transcoder")
			encoding.Sprites.Label = twitchtranscoder.SpritesDataMarker
			encoding.Sprites.FileBase = fmt.Sprintf(constTranscodePathSprites, basePath)

			log.Printf("Sprite generation is enabled %#v", encoding.Sprites)
		}
	}

	// Encoding
	encoding.Label = constTranscodeSource
	encoding.Path = fmt.Sprintf(constTranscodePathSource, basePath)
	encoding.PlaylistWindowSize = constTranscodeLivePlaylistWindowSize
	encoding.GenerateHlsPlaylist = constTwitchTranscoderGeneratePlaylist
	encoding.HlsValidation = constTranscodeHlsValidation
	encoding.QualityMetrics = constTranscodeQualityMetrics
	encoding.EncoderMetrics = constTranscodeEncoderMetrics
	encoding.Prefix = cfg.TwitchTranscoder.ChunkPrefix
	encoding.SegmentMinDuration = cfg.Segment.DurationRanges[cfg.Segment.TargetDuration].Min
	encoding.SegmentMaxDuration = cfg.Segment.DurationRanges[cfg.Segment.TargetDuration].Max
	encoding.EncoderType = cfg.TwitchTranscoder.EncoderType

	transcodeProfile := ChooseTranscoderProfile(cfg, profiles)
	log.Printf("Using Transcode Profile: %s\n", transcodeProfile)

	transcodeKeys := profiles.Profiles[transcodeProfile].TranscodeKeys
	_transcodes := make([]TranscodeQualitySettings, len(transcodeKeys))

	for index, profile := range transcodeKeys {
		_transcodes[index] = qualities.Qualities[profile]
		_transcodes[index].FileBase = fmt.Sprintf("%s/%s", basePath, _transcodes[index].Label)
		_transcodes[index].PlaylistGeneration = constTwitchTranscoderGeneratePlaylist
	}

	encoding.Transcodes = _transcodes

	return &encoding, nil
}
