package main

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

var (
	testBroadcastTag_1 = `broadcast_tag=2017TranscodeQS_V2&broadcaster=unknown_rtmp`
	testBroadcastTag_2 = `broadcast_tag=2017TranscodeX264_V2&recorder=0&broadcaster=watch_party`
	testBroadcastTag_3 = `broadcast_tag=2017TranscodeEvent_V2`
	testBroadcastTag_4 = `broadcaster=candybox`
)

func GetTestEncoderConfig() *EncoderConfig {
	return &EncoderConfig{
		Transcodes: []TranscodeQualitySettings{
			TranscodeQualitySettings{
				Label:   "high",
				Bitrate: 1600000,
				Height:  720,
				MaxFps:  60,
			},
			TranscodeQualitySettings{
				Label:   "high",
				Bitrate: 1600000,
				Height:  720,
				MaxFps:  30,
			},
			TranscodeQualitySettings{
				Label:   "high",
				Bitrate: 1600000,
				Height:  480,
				MaxFps:  30,
			},
		},
	}
}

func TestCullQuickSyncRenditionsFor30FpsInput(t *testing.T) {
	assert := assert.New(t)
	encoderConfigs := GetTestEncoderConfig()
	CullQuickSyncRenditions(encoderConfigs, 30)
	assert.Equal(2, len(encoderConfigs.Transcodes))

	for i := range encoderConfigs.Transcodes {
		if encoderConfigs.Transcodes[i].Height == 720 {
			if encoderConfigs.Transcodes[i].MaxFps != 30 {
				assert.Fail("Expected 60fps quicksync rendition to be removed")
			}
		}
	}
}

func TestCullQuickSyncRenditionsFor60FpsInputHighBitrate(t *testing.T) {
	assert := assert.New(t)
	encoderConfigs := GetTestEncoderConfig()
	CullQuickSyncRenditions(encoderConfigs, 60)
	assert.Equal(2, len(encoderConfigs.Transcodes))

	for i := range encoderConfigs.Transcodes {
		if encoderConfigs.Transcodes[i].Height == 720 {
			if encoderConfigs.Transcodes[i].MaxFps != 60 {
				assert.Fail("Expected 30fps quicksync rendition to be removed")
			}
		}
	}
}

//the input fps is low and we need to remove the higher fps
func TestCullRenditionsBasedOnFps(t *testing.T) {
	assert := assert.New(t)
	encoderConfigs := GetTestEncoderConfig()
	CullRenditionsBasedOnFps(encoderConfigs, 30)
	assert.Equal(2, len(encoderConfigs.Transcodes))

	for i := range encoderConfigs.Transcodes {
		if encoderConfigs.Transcodes[i].Height == 720 {
			if encoderConfigs.Transcodes[i].MaxFps == 60 {
				assert.Fail("Expected 60fps rendition to be removed")
			}
		}
	}
}

//the input fps is high and we keep all renditions lower than the input source as is
func TestCullRenditionsBasedOnHighInputFps(t *testing.T) {
	assert := assert.New(t)
	encoderConfigs := GetTestEncoderConfig()
	CullRenditionsBasedOnFps(encoderConfigs, 60)
	assert.Equal(3, len(encoderConfigs.Transcodes))
}

func TestOnly(t *testing.T) {
	assert := assert.New(t)
	encoderConfigs := GetTestEncoderConfig()
	CullRenditionsBasedOnFps(encoderConfigs, 30)
	CullQuickSyncRenditions(encoderConfigs, 30)
	assert.Equal(2, len(encoderConfigs.Transcodes))

}

func TestGetBroadcaster(t *testing.T) {
	assert := assert.New(t)
	*broadcastTag = testBroadcastTag_1
	assert.Equal("unknown_rtmp", GetBroadcaster())
	*broadcastTag = testBroadcastTag_2
	assert.Equal("watch_party", GetBroadcaster())
	*broadcastTag = testBroadcastTag_3
	assert.Equal("", GetBroadcaster())
	*broadcastTag = testBroadcastTag_4
	assert.Equal("candybox", GetBroadcaster())
}
