/*
Main Go Transcoder

Starts a twitch transcoding process on a channel
*/
package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"log"
	"net/url"
	"os"
	"os/signal"
	"path"
	"strconv"
	"strings"
	"syscall"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/config"
	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"
	"code.justin.tv/video/gotranscoder/pkg/noisy"
	"code.justin.tv/video/gotranscoder/pkg/notify"
	"code.justin.tv/video/gotranscoder/pkg/origin"
	"code.justin.tv/video/gotranscoder/pkg/pluginloop"
	"code.justin.tv/video/gotranscoder/pkg/spade"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/twitchtranscoder"
	"code.justin.tv/video/gotranscoder/pkg/usher"
	"code.justin.tv/video/gotranscoder/pkg/util"
	"code.justin.tv/video/gotranscoder/pkg/vod"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/protocols/hlsext"

	streamlog "code.justin.tv/video/streamlog/pkg/wswriter"
)

// Runtime configuration
var (
	channelPropertyID  int
	transcodeID        int
	listenerPort       int
	transcodesBasePath string
	transcodesPath     string
	transcodesFolder   string
	encoderConfigPath  string
	outputQualities    []string
	sessionID          string

	usherChannelStream *usher.StreamCodec
	usherTranscode     *usher.HlsTranscode
	stream             *usher.StreamInfo
	encoderConfig      *EncoderConfig
	onExit             *OnExit

	cfg                       config.Values
	es                        elasticsearch.Settings
	transcoder                twitchtranscoder.RuntimeSettings
	usherCfg                  usher.Settings
	segmentTrackingPlugin     SegmentHandler
	codecTrackingPlugin       CodecHandler
	streamlogPlugin           StreamlogHandler
	segmentPluginsContainer   pluginloop.ConcurrentPlugin
	codecPluginsContainer     pluginloop.ConcurrentPlugin
	thumbPluginsContainer     pluginloop.ConcurrentPlugin
	spritesPluginsContainer   pluginloop.ConcurrentPlugin
	adBreakPluginsContainer   pluginloop.ConcurrentPlugin
	streamlogPluginsContainer pluginloop.ConcurrentPlugin
	ProbeResult               avdata.ProbeResult
	bitrateUpdated            bool
	lvsMeta                   map[string]string
	notifier                  notify.Notifier
)

// Gotranscoder configuration flags
var (
	// gotrascoder specific flags
	configFile              = flag.String("gotranscoder_config", constGotranscoderConfig, "path to the gotrascoder configuration file")
	configProfiles          = flag.String("transcode_profiles", constSettingsTranscodeProfiles, "path to the Transcode stack profiles configuration file")
	configQualities         = flag.String("transcode_qualities", constSettingsTranscodeQualities, "path to the transcode quality settings configuration file")
	configTargetSegDuration = flag.String("target_segment_duration", "", "Override default target segment duration")
	statelessMode           = flag.Bool("stateless_mode", false, "This will cause GOT to run with keeping state tracking on external dependencies (for development and darklaunches)")
	vodPusherEnabled        = flag.Bool("vod_pusher_enabled", false, "Turns on gotranscoder's VOD pusher")
	transcodeVersion        = flag.Int("version", 1, "transcode type version. 1 = legacy and 2 = rivia. 1 is default")
	// job tracking keys

	nativeHeight        = flag.Int("native_height", 0, "source video height as best guessed by probe")
	overrideTranscodeID = flag.String("transcode_id", "", "TranscodeID in Usher - if this is not set - it will be queried from user. Should be set in production")
	transcodeProfile    = flag.String("transcode_profile", "", "override the transcode profile defined in the config")
	jobKey              = flag.String("job_key", "", "key for the job created by usher. Used to track this transcode.")
	channel             = flag.String("channel", "", "streamer channel to parse")
	channelID           = flag.Uint64("channel_id", 0, "streamer's channel_id in siteDB")
	rtmpEndpoint        = flag.String("rtmp_endpoint", "", "rtmp endpoint, like localhost:1935/app")
	rtmpUrl             = flag.String("rtmp_url", "", "full rtmp url, like localhost:1935/app/live_user_<channelname>")
	broadcastTag        = flag.String("broadcast_tag", "", "query params at end of streamkey, like videoKeyframeFrequency=5&totalDatarate=200. somewhat confusingly, we use a query param named broadcast_tag to use debug transcode profiles defined in config/transcode_profiles.json like -broadcast_tag 'broadcast_tag=Coral1080Test'")
	useOrigin           = flag.Bool("use_origin", false, "Flag which specifies that we have to use origin")
	drm                 = flag.Bool("drm", false, "Flag which specifies whether encryption is enabled or not")
	shadowSuffix        = flag.String("shadow_suffix", "", "Tells the gotranscoder to operate in a different directory from normal, one suffixed by 'shadow_suffix'")
	// Flags below are introduced for LVS Streams
	s3Bucket     = flag.String("s3_bucket", "", "The S3 bucket to be used to record Vods")
	s3Prefix     = flag.String("s3_prefix", "", "The S3 path in the bucket to be used to store Vods")
	snsEndpoint  = flag.String("sns_endpoint", "", "The SNS Endpoint to be used to send notifications for streams")
	disableVinyl = flag.Bool("disable_vinyl", false, "Disable VOD registration to Vinyl")
	environment  = flag.String("env", "", "The environment in which this service runs")
	cdnUrl       = flag.String("cdn_url", "", "The cdn domain url to be used for recording VOD manifests")
	fileFormat   = flag.String("file_format", "ts", "Extension of the segment file")
)

// Define which TwitchTranscoder to Uses
func initTwitchTranscoderSelector() string {
	if os.Getenv(constIntelSDKEnvVar) == constPatchedSDKVersion {
		statsd.Inc(constStatsTTPatched, 1, 1)
		log.Println("Selecting transcoder patched version")
		return cfg.TwitchTranscoder.PathPatched
	}
	if os.Getenv(constBetaTranscoderOverrideEnvVar) == "true" {
		statsd.Inc(constStatsTTBeta, 1, 1)
		log.Println("BETA transcoder overriding env var found and set to true, using Beta twitchtranscoder")
		return cfg.TwitchTranscoder.PathBeta
	}

	response, err := usherCfg.GetDbOption(constDbOptionTranscoderBetaRatio)
	if err != nil {
		log.Println("Failed to get the twitch transcoder beta ratio - defaulting to Stable")
		statsd.Inc(constStatsTTStable, 1, 1)
		return cfg.TwitchTranscoder.PathStable
	}

	consistentHash := util.ConsistentHash(*channel, constConsistentHashPrecision)
	if usherBetaRatio, ok := response.Value.(float64); ok {
		log.Printf("Usher beta ratio = %e\nHashed channel(%s) score = %e", usherBetaRatio, *channel, consistentHash)
		if usherBetaRatio-constFloatTolerance < 0.0 {
			log.Println("Usher beta ratio = 0.0, use TwitchTranscoder Stable")
			statsd.Inc(constStatsTTStable, 1, 1)
			return cfg.TwitchTranscoder.PathStable
		} else if usherBetaRatio+constFloatTolerance > 1.0 {
			log.Println("Usher beta ratio = 1.0, use TwitchTranscoder Beta")
			statsd.Inc(constStatsTTBeta, 1, 1)
			return cfg.TwitchTranscoder.PathBeta
		}
		if usherBetaRatio >= consistentHash {
			log.Println("Channel consistent hash falls under the beta threshold: use TwitchTranscoder Beta")
			statsd.Inc(constStatsTTBeta, 1, 1)
			return cfg.TwitchTranscoder.PathBeta
		}
		log.Println("Channel consistent hash doesn't fall under the beta threshold: use TwitchTranscoder Stable")
		statsd.Inc(constStatsTTStable, 1, 1)
		return cfg.TwitchTranscoder.PathStable

	}
	log.Printf("Error: Usher beta ratio is not of type float64 but %T, use TwitchTranscoder Stable", response.Value)
	statsd.Inc(constStatsTTStable, 1, 1)
	return cfg.TwitchTranscoder.PathStable
}

// This method will get the selected transcode profile and generate an encoder config based on it.
// Today transcode profiles are separate from Chunked (transcode) and audoi_only, which get appended no matter what.
func initEncoderConfig(transcode *usher.HlsTranscode, transcodesPath string, probeResult *avdata.ProbeResult) (*EncoderConfig, error) {

	fps := int(probeResult.ProbeAV.Video.Fps)

	if fps == 0 {
		fps = defaultVideoFps
	}

	fps = roundOffFps(fps)

	inputHeight := int32(probeResult.ProbeAV.Video.Height)
	log.Printf("[PROBE] Probe Output: %+v", ProbeResult)

	//Add statsd metric and log channel name for number of channels with low bitrate for 720p60 and 1080p60
	if fps == defaultVideoFps && (inputHeight == const720pVideoHeight || inputHeight == const1080pVideoHeight) {
		log.Println("[PROBE] Detected low bitrate at high resolution from probe data:", *channel)
		statsd.Inc(constLowBitrateHighResolution, 1, 1)
	}
	if _, ok := cfg.Segment.DurationRanges[*configTargetSegDuration]; ok {
		cfg.Segment.TargetDuration = *configTargetSegDuration
	} else {
		log.Printf("TargetSegmentDuration %q is not valid. Using default %q",
			*configTargetSegDuration, cfg.Segment.TargetDuration)
	}

	encoder, err := GenerateEncoderConfig(&cfg, transcodesPath)
	if err != nil {
		return nil, err
	}

	rtmpUrl := getRtmpURL()
	log.Printf("Connecting to RTMP stream @ %s", rtmpUrl)
	encoder.RtmpURL = rtmpUrl

	// twitchtranscoder needs broadcaster info for detailed warning msg
	encoder.Broadcaster = GetBroadcaster()

	// Get the segment duration for the usher
	var pushersInfo map[string]interface{}
	if err2 := json.Unmarshal([]byte(transcode.PushersInfo), &pushersInfo); err2 == nil {
		log.Printf("PUSHERS INFO : %+v\n", pushersInfo)
	}

	//Cull Renditions based on height and framerate only for rivia stacks
	if int(*transcodeVersion) == riviaVersion {
		//Cull renditions based on probe video height
		CullRedundantTranscodes(encoder, inputHeight)
		//Cull Renditions based on fps
		CullRenditionsBasedOnFps(encoder, fps)
		if cfg.TwitchTranscoder.EncoderType == quickSyncEncoderType {
			CullQuickSyncRenditions(encoder, fps)
		}
	}

	maxIdrInterval := int(probeResult.ProbeAV.Video.MaxIdrInterval)
	// Set min and max segment duration for long gop channels
	if maxIdrInterval > constMaxSegmentDuration {
		if len(encoder.Transcodes) > 0 {
			encoder.SegmentMinDuration = constAverageSegmentDuration
			encoder.SegmentMaxDuration = constAverageSegmentDuration
		} else {
			encoder.SegmentMinDuration = constMinSegmentDuration
			encoder.SegmentMaxDuration = constMaxSegmentDuration
		}
	}

	return encoder, err
}

func getRtmpURL() string {
	if *rtmpUrl != "" {
		return *rtmpUrl
	}

	if !strings.HasPrefix(*channel, lowLatencyDarkLaunchPrefix) {
		return fmt.Sprintf("%s/live_user_%s", *rtmpEndpoint, *channel)
	}

	channelName := strings.TrimPrefix(*channel, lowLatencyDarkLaunchPrefix)
	return fmt.Sprintf("%s/live_user_%s", *rtmpEndpoint, channelName)
}

func initStreamPropertyID() (*usher.StreamCodec, error) {
	channelProperties, err := usherCfg.GetStreamCodec(*channel)
	if err != nil {
		return nil, err
	}

	channelPropertyID = channelProperties.ID
	log.Printf("CHANNEL PROP ID %d\n", channelPropertyID)
	return channelProperties, err
}

func initTranscodeEntry() (*usher.HlsTranscode, error) {
	if len(*overrideTranscodeID) > 0 {
		id, err := strconv.Atoi(*overrideTranscodeID)
		if err != nil {
			return nil, err
		}

		transcode, err := usherCfg.GetHlsTranscode(id)
		if err != nil {
			return nil, err
		}
		return transcode, nil
	}

	// TODO: get usher transcode entry by channel_id when API is available

	transcode, err := usherCfg.GetShowChannel(*channel)
	if err != nil {
		return nil, err
	}

	return transcode, nil
}

// initTranscodeQualities will generate an array of the selected qualities for this broadcast.
// qualities are defined based on a transcode profile selected via a the profile speficied in the command line arguments
// or an override via rtmp parameter. This profile as defined in transcodeQualities and transcodeProfiles are the transcode
// stack configurations of how the stream will be setup.
func initTranscodeQualities(encoder *EncoderConfig) {
	outputQualities = make([]string, 0, (len(encoder.Transcodes) + 2))
	outputQualities = append(outputQualities, constTranscodeSource)
	for _, transcodeEntry := range encoder.Transcodes {
		outputQualities = append(outputQualities, transcodeEntry.Label)
	}
	outputQualities = append(outputQualities, constTranscodeAudioOnly)

}

func initTranscodeDirectories(basePath string, encoder *EncoderConfig) (string, error) {
	// Create the base directory
	err := os.MkdirAll(basePath, 0755)
	if err != nil {
		return "", fmt.Errorf("Error creating transcode main output path '%s' - %s", basePath, err)
	}

	// Write the config in the base directory
	path, err := WriteEncoderConfig(encoder, basePath)
	if err != nil {
		return "", fmt.Errorf("Error creating encoder config '%s' - %s ", path, err)
	}

	// Build a manifest of directories that we need to create
	dirs := []string{
		fmt.Sprintf(constTranscodePathAudio, basePath),
		fmt.Sprintf(constTranscodePathThumb, basePath),
		fmt.Sprintf(constTranscodePathSource, basePath),
		fmt.Sprintf(constTranscodePathSprites, basePath),
	}
	for _, transcodeEntry := range encoder.Transcodes {
		dirs = append(dirs, transcodeEntry.FileBase)
	}

	// Create all the directories, return an error on the first failure
	for _, dir := range dirs {
		log.Printf("Creating output directory '%s'", dir)
		err := os.MkdirAll(dir, 0755)
		if err != nil {
			return "", fmt.Errorf("Error creating output directory '%s' - %s", dir, err)
		}
	}

	return path, nil
}

func registerPlugins(transcodeSegmentsPath string, hlsUrlBase string, usherTrans *usher.HlsTranscode, encoder *EncoderConfig, notifier notify.Notifier, probeResult *avdata.ProbeResult, transcodeURI string) {
	// Create a plugin container for segments
	segmentPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	// Create plugin container for codecs
	codecPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	// Create plugin container for thumbnails
	thumbPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	// Create plugin container for sprites
	spritesPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	// Create plugin container for weaver AdBreakRequests
	adBreakPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	//Create plugin container for Streamlog
	streamlogPluginsContainer = pluginloop.ConcurrentPlugin{
		EvalProcessingTimeout: constPluginProcessTimeout,
	}

	// Don't block when receiving ad break messages. We would block if
	// we have trouble connecting to pubsub or subscribing to the
	// topic. By making this async, there's a risk that we miss some ad
	// insertion messages, but in exchange startup is faster, and the
	// risk of dropped messages is low because we should connect very
	// quickly if pubsub is healthy. If pubsub is unhealthy, we would
	// miss the messages anyway.
	go func() {
		if err := receiveAdBreakMessages(); err != nil {
			// This error shouldn't be fatal. Just log it.
			log.Printf("error setting up ad insertion listener, not going to insert ads for this session: %s", err)
		}
	}()

	// initialize the segment processing plugin and register
	segmentTrackingPlugin.SegmentPath = transcodeSegmentsPath
	segmentTrackingPlugin.transcodeURI = transcodeURI
	segmentTrackingPlugin.SegmentWindow = constTranscodeLivePlaylistWindowSize
	segmentTrackingPlugin.TranscodeID = transcodeID
	segmentTrackingPlugin.TargetDuration = cfg.Segment.DurationRanges[cfg.Segment.TargetDuration].Max
	segmentTrackingPlugin.usher = &usherCfg
	segmentTrackingPlugin.transcodeUsherUpdateInterval = constTranscodeUsherUpdateMillis * time.Millisecond
	segmentTrackingPlugin.usherUpdateStreamInterval = constUsherUpdateStreamMillis * time.Millisecond
	if *fileFormat == fileFormatFmp4 {
		segmentTrackingPlugin.FileFormat = hlsext.ExtendedPlaylist_MP4
	}
	segmentTrackingPlugin.Initialize()
	if err := segmentPluginsContainer.Register(&segmentTrackingPlugin); err != nil {
		log.Fatalf("Error registering segment plugin on segmentPluginsContainer: Exiting: %s", err)
	}
	if err := adBreakPluginsContainer.Register(&segmentTrackingPlugin); err != nil {
		log.Printf("Error registering segment plugin on adBreakPluginsContainer: Ignoring: %s", err)
	}

	// Register codec tracker
	if err := codecPluginsContainer.Register(codecTrackingPlugin); err != nil {
		log.Fatalf("Error registering codec plugins: Exiting %s", err)
	}

	// TODO: Register weaver playlist writers for each encoding

	// Register vod pusher if enabled
	if *vodPusherEnabled {
		*cdnUrl = strings.TrimSuffix(*cdnUrl, "/")
		vodPusher := initVODPusher(&cfg, usherTrans.Destination, encoder, hlsUrlBase, notifier, probeResult)
		if *s3Bucket != "" {
			lvsMeta[lvsS3VodUrl] = getS3VodUrl(strings.TrimSuffix(vodPusher.Bucket, "/"), vodPusher.VodKey)
			//Write manifest vod url
			if vodPusher.VodDistributionUrl != "" {
				lvsMeta[lvsVodManifestUrl] = fmt.Sprintf("%s/%s/%s", vodPusher.VodDistributionUrl, vodPusher.VodKey, manifestPath)

				go GenerateAndWriteVODManifest(vodPusher)
			}
		}
		if err := segmentPluginsContainer.Register(vodPusher); err != nil {
			log.Fatalln("Failed to register vodpusher plugin", err)
		}
		if err := codecPluginsContainer.Register(vodPusher); err != nil {
			log.Fatalln("Failed to register vodpusher plugin", err)
		}
		if err := thumbPluginsContainer.Register(vodPusher); err != nil {
			log.Fatalln("Failed to register vodpusher plugin", err)
		}
		if err := spritesPluginsContainer.Register(vodPusher); err != nil {
			log.Fatalln("Failed to register vodpusher plugin", err)
		}
	}

	// Register StreamlogHandler in Segment, Codec and Streamlog plugin containers
	// Only log a Warning message instead of exiting so that Transcode can continue
	if !*statelessMode {
		if err := segmentPluginsContainer.Register(&streamlogPlugin); err != nil {
			log.Printf("Error registering Streamlog plugin on segmentPluginsContainer: Exiting: %s", err)
		}
		if err := codecPluginsContainer.Register(&streamlogPlugin); err != nil {
			log.Printf("Error registering Streamlog plugin on codecPluginsContainer: Exiting: %s", err)
		}
		if err := streamlogPluginsContainer.Register(&streamlogPlugin); err != nil {
			log.Printf("Error registering Streamlog plugin on streamlogPluginsContainer: Exiting: %s", err)
		}
	}
}

func prepareStreamInUsher(transcodeID int, listenerPort int, transcodeURI string) {
	if *statelessMode {
		log.Println("[STATELESS MODE] skipping prepareStreamInUsher")
		return
	}
	var transcodeEntry usher.HlsTranscode
	// Do an emptry stream info update at the beginning of the stream.
	// Marks it as not ready
	transcodeEntry.Status = "pending"
	transcodeEntry.Version = int(*transcodeVersion)
	transcodeEntry.JobKey, _ = strconv.Atoi(*jobKey)
	transcodeEntry.ID = transcodeID
	transcodeEntry.Channel = *channel
	transcodeEntry.ChannelID = *channelID
	transcodeEntry.TranscoderInfo = fmt.Sprintf("gotranscoderport=%d", listenerPort)
	transcodeEntry.OriginFQDN = osFQDN()
	transcodeEntry.TranscodeProfile = *transcodeProfile
	transcodeEntry.URI = transcodeURI

	meta, err := RenditionMetadata(encoderConfig, ProbeResult)
	if err != nil {
		log.Println("Error in generating rendition metadata", err)
	}

	jsonMeta, err := json.Marshal(meta)
	if err != nil {
		log.Println("Error in marshalling rendition metadata", err)
	}
	transcodeEntry.RenditionMetadata = string(jsonMeta)
	//Also update the LVSMetadata
	if len(lvsMeta) != 0 {
		lvsJsonMeta, err := json.Marshal(lvsMeta)
		if err != nil {
			log.Println("Error in marshalling LVS Metadata", err)
		} else {
			transcodeEntry.LvsMetadata = string(lvsJsonMeta)
		}
	}

	// Log transcode stack version to Streamlog to be processed by Inspector
	streamlog.Log(*channel, "transcode_stack_version", int(*transcodeVersion))
	log.Printf("Setting up transcode ID %d in Usher: %+v\n", transcodeID, transcodeEntry)
	if err := usherCfg.UpdateHlsEntry(transcodeID, transcodeEntry); err != nil {
		log.Println("Failed to update usher hls entry", err)
	}
}

// Launch transcoder process
func main() {
	flag.Parse()
	log.SetFlags(log.Ltime | log.Lshortfile)
	log.SetPrefix("\033[35;1mGT:\033[39;1m")
	log.Println("Starting GoTranscoder")

	runtimeStart := time.Now()

	// Create OnExit handler. Callbacks will be run when we cleanly exit
	onExit = new(OnExit)
	defer onExit.Cleanup()

	// Handle signals
	go func() {
		c := make(chan os.Signal, 1)
		signal.Notify(c, os.Interrupt)
		signal.Notify(c, syscall.SIGTERM)
		<-c
		statsd.Inc(constStatsSignalBrakeExit, 1, 1)
		log.Println("CTRL+C received - Exiting")
		go func() {
			<-c // Exit if we get another signal (cleanup hangs)
			os.Exit(1)
		}()
		onExit.Cleanup()
		os.Exit(1)
	}()

	var err error

	// Initialization steps
	if err = config.Load(*configFile, &cfg); err != nil {
		log.Fatalf("Error loading gotranscoder configuration - exiting: %s", err)
	}

	usherCfg = usher.Settings{
		BaseURL:   cfg.Usher.Host,
		Stateless: *statelessMode,
	}

	// cleanup entry after exit - always run this
	onExit.Add(func() {
		segmentTrackingPlugin.FinalPlaylist = true
		segmentTrackingPlugin.ForcePlaylistFlush()
		time.Sleep(4 * time.Second)

		// Stateless mode : Do not update state tracking
		if *statelessMode {
			log.Println("[STATELESS MODE] skipping usher transcode removal and finalization")
			return
		}

		log.Println("Issuing Transcode entry removal on Usher")
		err = usherCfg.KillHlsTranscode(usherTranscode.ID)
		if err != nil {
			log.Printf("Error killing transcode in usher after exit %s", err)
		}

		statsd.Inc(constStatsCleanExit, 1, 1)
		statsd.Timing(constStatsExecutionTime, time.Since(runtimeStart), 1.0)

	})

	// If no override's are specified via command line flags.  set them up from usher properties
	if len(*rtmpEndpoint) <= 0 {
		log.Println("Using stream rtmpEndpoint:", stream.Connect)
		*rtmpEndpoint = stream.Connect
	}

	log.Println("broadcastTag:", *broadcastTag)
	if len(*broadcastTag) <= 0 && stream != nil {
		log.Println("Using stream broadcastTag:", stream.BroadcastTag)
		*broadcastTag = stream.BroadcastTag
	}

	//Override the vod pusher flag based on the recorder flah in broadcast tags
	if len(*broadcastTag) > 0 {
		streamParams, _ := url.ParseQuery(*broadcastTag)
		tags := streamParams["recorder"]

		if len(tags) > 0 {
			value := strings.ToLower(tags[0])
			if value == "0" || value == "false" {
				*vodPusherEnabled = false
				log.Println("Vod Pusher is disabled based on override from broadcast_tags")
			} else if value == "1" || value == "true" {
				*vodPusherEnabled = true
				log.Println("Vod Pusher is enabled based on override from broadcast_tags")
			}
		}
	}

	// Get transcode ID by querying the channel in usher.
	if usherTranscode, err = initTranscodeEntry(); err != nil {
		log.Fatalf("Error retrieving HLS transcode settings from usher.  exiting: %s, %v", err, usherTranscode)
	}

	transcodeID = usherTranscode.ID
	sessionID = usherTranscode.SessionID

	log.Printf("Session ID: %s", sessionID)
	log.Printf("Channel ID: %d", usherTranscode.ChannelID)

	//TODO: Remove once we move to channel_id completely
	if *channelID == 0 {
		*channelID = usherTranscode.ChannelID
	}

	if usherChannelStream, err = initStreamPropertyID(); err != nil {
		log.Fatalf("Error retrieving channel stream properties from usher.  exiting: %s, %v", err, usherChannelStream)
	}

	// Settings
	stats := statsd.Settings{
		Enabled:     cfg.Statsd.Enabled,
		Port:        cfg.Statsd.Port,
		Host:        cfg.Statsd.Host,
		ServiceName: cfg.Statsd.ServiceName,
		LVS:         isLvsChannel(),
		ChannelName: *channel,
		Env:         *environment,
	}

	// Graphite stats
	stats.Connect()
	// Initialize local directories and encoder config
	ttPath := initTwitchTranscoderSelector()
	// TODO: This should be changed so that this directory is used as passed by the workers rather than
	// 		reconstructed a second time.
	transcodesFolder = fmt.Sprintf("%s_%d_%s%s", *channel, usherTranscode.Destination, *jobKey, *shadowSuffix)
	transcodesBasePath = fmt.Sprintf("%s/%s", cfg.Archives.Path, transcodesFolder)

	log.Println("[PROBE] running stream probe")
	// Probe channel and get probe results from Twitch Transcoder
	twitchtranscoder.SetProbeTimeoutDuration(13 * time.Second)
	probeRes := twitchtranscoder.RunTwitchProbe(getRtmpURL(), ttPath)

	//Record how much time it took from starting the gotranscoder process till we finish probing
	statsd.Timing(constTimeInitToProbe, time.Since(runtimeStart), 1.0)

	if probeRes == "" {
		statsd.Inc(probeFailCount, 1, 1)
		log.Println("[PROBE]No output received from Twitch Transcoder probe")
	}

	res, probeFailed := twitchtranscoder.ParseProbeOutput(probeRes)
	if probeFailed {
		//Set Video Defaults
		res.ProbeAV.Video.Height = defaultVideoheight
		res.ProbeAV.Video.Width = defaultVideowidth
		res.ProbeAV.Video.Bitrate = defaultVideoBitRate
		res.ProbeAV.Video.Codec = defaultVideoCodec
		res.ProbeAV.Video.Fps = defaultVideoFps
		res.ProbeAV.Video.MaxIdrInterval = defaultMaxIdrInterval
		//Set audio Defaults
		res.ProbeAV.Audio.Codec = defaultAudioCodec
		res.ProbeAV.Audio.Bitrate = defaultAudioBitRate

	}

	ProbeResult = res

	if encoderConfig, err = initEncoderConfig(usherTranscode, transcodesBasePath, &ProbeResult); err != nil {
		log.Fatalf("Error generating encoder config- exiting: %s, %+v", err, encoderConfig)
	}
	if transcodesPath, err = initTranscodeDirectories(transcodesBasePath, encoderConfig); err != nil {
		log.Fatalf("Error generating transcode subdirectories - exiting: %s, %s", err, transcodesPath)
	}

	// If origin is enabled, configure TwitchTranscoder to
	// 1. disable playlist generation
	// 2. disable segment writing to /dev/shm/archives
	if *useOrigin {
		encoderConfig.Path = ""
		encoderConfig.Thumbs.Path = ""
		encoderConfig.Audio.Path = ""
		for k, v := range encoderConfig.Transcodes {
			v.FileBase = ""
			v.PlaylistGeneration = false
			encoderConfig.Transcodes[k] = v
		}
	}

	if encoderConfigPath, err = WriteEncoderConfig(encoderConfig, transcodesBasePath); err != nil {
		log.Fatalf("Error writing encoder config to disk - exiting: %s, %s", err, encoderConfigPath)
	}
	initTranscodeQualities(encoderConfig)
	listenerPort, err = initHTTP()
	if err != nil {
		log.Fatalf("Error initializing http listener - exiting: %s", err)
	}

	// Init Streamlog
	slcfg := streamlog.Settings{
		WebsocketEndpoint: cfg.Streamlog.Endpoint,
		SessionID:         sessionID,
		ServiceTag:        cfg.Streamlog.ServiceTag,
		Proxy:             cfg.Streamlog.Proxy,
		QueueSize:         cfg.Streamlog.QueueSize,
	}
	streamlog.Init(slcfg)

	// Initialize low-latency origin session
	var (
		hlsUrlBase         string
		originSession      *originctl.Session
		originExitCallback func()
		originEndpoint     string
	)
	if *useOrigin {
		log.Println("Attempting to initialize origin session")
		originEndpoint = cfg.Origin.OriginEndpoint
		originSession, originExitCallback, err = initOriginSession(originEndpoint, *channelID, outputQualities)
		if err != nil {
			streamlog.Log(*channel, "step_gotranscoder_origin_connect", false)
			log.Println("Error initializing origin session:", err)
		} else {
			streamlog.Log(*channel, "step_gotranscoder_origin_connect", true)
			hlsUrlBase = origin.HLSURLBase(originEndpoint, originSession)
		}
	}

	transcoder = twitchtranscoder.RuntimeSettings{
		Channel:            *channel,
		EncoderConfig:      encoderConfigPath,
		Path:               ttPath,
		HlsUrlBase:         hlsUrlBase,
		OutputJSONParser:   ParseJSONLine,
		OutputDebugParser:  ParseDebugLine,
		ListenerPort:       listenerPort,
		DisableObfuscation: *shadowSuffix != "",
		FileExtension:      *fileFormat,
	}

	log.Printf("Gotranscoder running TT with transcode Config: %+v", transcoder)

	initElasticSearch()
	go LogHostConfiguration()

	noisy.Configure(&noisy.Config{
		ESClient:         &es,
		RollbarToken:     cfg.Noisy.RollbarToken,
		TranscodeProfile: *transcodeProfile,
		Channel:          *channel,
		ChannelID:        *channelID,
	}, "localhost:8500")

	// setup SNS notifier
	var customerId, contentId string
	if isLvsChannel() {
		customerId, contentId = getLvsChannelParts()
	} else {
		*snsEndpoint = cfg.Notifications.StreamEventTopicARN
	}
	notifier = notify.InitNotifier(*channel, int(*channelID), usherTranscode.Destination, sessionID, *broadcastTag, customerId, contentId, *snsEndpoint, isLvsChannel(), &es)

	go func() {
		//Initialize spade client and send transcode stream start event
		spadeclient, err := spade.GetSpadeClient(cfg.Spade.BaseUrl, cfg.Spade.Scheme)
		if err != nil {
			log.Printf("error in initializing spade client :%v", err)
		} else {
			spadeclient.SendStreamStartEvent(*channel, customerId, contentId, sessionID, *transcodeProfile, *channelID, runtimeStart.Unix(), int64(usherTranscode.Destination), *vodPusherEnabled)
		}
	}()

	// Transcode Process
	// starts TwitchTranscoder, regiter  m3u8 and segment plugins, mark stream as ready.
	statsd.Inc(constStatsGotStartTime, 1, 1)
	lvsMeta = make(map[string]string)
	if *snsEndpoint != "" {
		lvsMeta[lvsSNSEndpoint] = *snsEndpoint
	}

	//Determine Transcode URI
	transcodeURI, err := getTranscodeURI(originSession, transcodesFolder)
	if err != nil {
		log.Fatalf("Error in getting transcode URI %v", err)
	}

	registerPlugins(transcodesBasePath, hlsUrlBase, usherTranscode, encoderConfig, notifier, &ProbeResult, transcodeURI)
	prepareStreamInUsher(transcodeID, listenerPort, transcodeURI)

	// Do this after registering VOD Pusher plugin
	if originExitCallback != nil {
		onExit.Add(originExitCallback)
	}

	// The origin server also wants to receive tenfoot-style data about playlists.
	// This must be called after initTenfoot because initOriginFoot wraps
	// segmentTrackingPlugin.tenfootGlue.
	origSettings := &originSettings{
		channel:          *channel,
		originEnabled:    *useOrigin,
		endpoint:         originEndpoint,
		transcodeProfile: *transcodeProfile,
		resourcePath:     transcodesFolder,
		variants:         outputQualities,
		usherStreamId:    transcodeID,
		sessionId:        sessionID,
	}
	if *fileFormat == fileFormatFmp4 {
		origSettings.fileFormat = hlsext.ExtendedPlaylist_MP4
	}

	err = initOriginFoot(&cfg, &segmentTrackingPlugin, originSession, runtimeStart, origSettings)
	if err != nil {
		log.Printf("error while initializing originfoot client: %v", err)
	} else {
		log.Printf("sending tenfoot playlists to originfoot too")
	}

	// add timing statsd
	timeToJobStart := time.Now().Sub(time.Unix(int64(usherTranscode.StartedOn), 0))
	statsd.Timing(constTimeToTranscode, timeToJobStart, 1.0)

	timeToInitialize := runtimeStart.Sub(time.Unix(int64(usherTranscode.StartedOn), 0))
	statsd.Timing(constTimeToInitialized, timeToInitialize, 1.0)

	go ReadyLoop(transcodesBasePath, outputQualities, notifier)
	// Run the actual transcoder as a subprocess.
	err = transcoder.Run()
	if err != nil {
		log.Printf("Error starting TwitchTranscoder %s", err)
		statsd.Inc(constStatsUnexpectedExit, 1, 1)
	}
}

func getTranscodeURI(originSession *originctl.Session, transcodeFolder string) (string, error) {
	host, err := os.Hostname()
	if err != nil {
		log.Println("Unable to dectect hostname, exit")
		return "", err
	}

	if originSession != nil {
		return path.Join("/", host, "lowlatency-darklaunch", "streams", originSession.Id), nil
	}
	return path.Join("/new", host, transcodeFolder), nil

}

func isLvsChannel() bool {
	if strings.HasPrefix(*channel, "lvs.") && len(strings.Split(*channel, ".")) == 3 {
		return true
	}
	return false
}

func getLvsChannelParts() (customerId, contentId string) {
	if !isLvsChannel() {
		return
	}
	parts := strings.Split(*channel, ".")
	customerId = parts[1]
	contentId = parts[2]
	return
}

func getS3VodUrl(bucket string, key string) string {
	return fmt.Sprintf(lvsS3VodUrlFormat, bucket, key)
}

// GenerateAndWriteVODManifest generates a static vod manifest from the rendition data and stores it in S3
func GenerateAndWriteVODManifest(pusher *vod.Pusher) {
	//Generate static vod manifest and write to S3 at the beginning
	meta, err := RenditionMetadata(encoderConfig, ProbeResult)
	if err == nil {
		data := vod.GenerateVODManifest(meta, pusher.VodDistributionUrl, pusher.VodKey)
		// Push the url in lvsMeta to make it visible in GetStream and ListStream API's
		pusher.ProcessVodManifest(data, manifestPath)
	}
}
