package main

import (
	"fmt"
	"log"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/usher"
)

// RenditionMetadata creates the metadata around mux rate and qualities for master manifest generation
func RenditionMetadata(cfg *EncoderConfig, probeData avdata.ProbeResult) ([]usher.MuxInfo, error) {
	return RenditionMetadataFromTranscodes(cfg.Transcodes, probeData)
}

// RenditionMetadataFromTranscodes pulls rendition metadata from the probe results for use in
// determining appropriate encoder configurations and storage in our central db
func RenditionMetadataFromTranscodes(transcodes []TranscodeQualitySettings, probeData avdata.ProbeResult) ([]usher.MuxInfo, error) {
	muxData := make([]usher.MuxInfo, 0)
	//setup the variables to be used for generating muxdata
	var height = int(probeData.ProbeAV.Video.Height)
	var width = int(probeData.ProbeAV.Video.Width)
	var fps = roundOffFps(int(probeData.ProbeAV.Video.Fps))
	var codecs = probeData.ProbeAV.Video.Codec + "," + probeData.ProbeAV.Audio.Codec
	var maxIdrInterval = int64(probeData.ProbeAV.Video.MaxIdrInterval)
	var muxrate = calculateMuxRate(int(probeData.ProbeAV.Audio.Bitrate), int(probeData.ProbeAV.Video.Bitrate))
	var audioMux = int(probeData.ProbeAV.Audio.Bitrate)
	var audioCodec = probeData.ProbeAV.Audio.Codec
	var displayLabel string

	// source
	if fps == 60 {
		displayLabel = fmt.Sprintf("%dp%d (source)", height, fps)
	} else {
		displayLabel = fmt.Sprintf("%dp (source)", height)
	}
	muxData = append(muxData, usher.MuxInfo{
		GroupID:        "chunked",
		DisplayLabel:   displayLabel,
		MuxRate:        muxrate,
		VideoHeight:    height,
		VideoWidth:     width,
		Codecs:         codecs,
		Fps:            fps,
		MaxIdrInterval: maxIdrInterval,
	})

	// Transcodes
	for _, v := range transcodes {
		log.Printf("[MUX] found: %s", v.Label)
		if v.Height == 720 && v.MaxFps == 60 {
			displayLabel = fmt.Sprintf("%dp%d", v.Height, v.MaxFps)
		} else {
			displayLabel = fmt.Sprintf("%dp", v.Height)
		}

		if v.Muxrate > 0 {
			muxrate = int(v.Muxrate)
		} else {
			muxrate = calculateMuxRate(audioMux, int(v.Bitrate))
		}

		muxData = append(muxData, usher.MuxInfo{
			GroupID:      fmt.Sprintf("%dp%d", v.Height, v.MaxFps),
			DisplayLabel: displayLabel,
			MuxRate:      muxrate,
			VideoHeight:  int(v.Height),
			VideoWidth:   calculateVideoWidth(int(v.Height), height, width),
			Codecs:       v.CodecString,
			Fps:          int(v.MaxFps),
		})

	}

	//audio
	muxData = append(muxData, usher.MuxInfo{
		GroupID:      "audio_only",
		DisplayLabel: fmt.Sprintf("audio_only"),
		MuxRate:      audioMux,
		VideoHeight:  0,
		VideoWidth:   0,
		Codecs:       audioCodec,
		Fps:          0,
	})

	return muxData, nil
}

func calculateVideoWidth(videoHeight int, probeVideoHeight int, probeVideoWidth int) int {
	if probeVideoHeight == 0 {
		return 0
	}
	vw := int((videoHeight * probeVideoWidth) / probeVideoHeight)
	return (vw & constVideoWithCalculatorMask)
}

func calculateMuxRate(probeAudioBitrate int, videoBitRate int) int {
	if probeAudioBitrate == 0 {
		return int(float32(videoBitRate+defaultAudioBitRate) * constTransportCostMultiplier)
	}

	return int(float32(videoBitRate+probeAudioBitrate) * constTransportCostMultiplier)
}

func roundOffFps(fps int) int {
	standardFps := [6]int{10, 20, 24, 30, 48, 60}
	var res int
	var minDiff = 1000
	for _, element := range standardFps {
		if fps > element {
			if fps-element < minDiff {
				minDiff = fps - element
				res = element
			}
		} else {
			if element-fps < minDiff {
				minDiff = element - fps
				res = element
			}
		}
	}
	return res
}
