package main

import (
	"testing"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"github.com/stretchr/testify/assert"
)

var (
	muxGenTestEncoderConfig = &EncoderConfig{
		Transcodes: []TranscodeQualitySettings{
			TranscodeQualitySettings{
				Label:       "high",
				Bitrate:     1600000,
				DisplayName: "720p60_3100k",
				MaxFps:      60,
				Height:      720,
				CodecString: "avc1.77.31,mp4a.40.2",
			},
			{
				Label:         "mobile",
				Bitrate:       180000,
				DisplayName:   "160p",
				MaxFps:        30,
				Height:        160,
				CodecString:   "avc1.77.31,mp4a.40.2",
				Muxrate:       230000,
				AudioBitrate:  24000,
				AudioChannels: 2,
			},
		},
	}
)

func TestRenditionMetadataWithProbeData(t *testing.T) {
	assert := assert.New(t)
	var probeData = avdata.ProbeResult{
		ProbeAV: avdata.ProbeAV{
			Video: avdata.ProbeVideo{
				Bitrate:        1000,
				Codec:          "test_codec",
				Profile:        "test_profile",
				Height:         100,
				Width:          200,
				Fps:            60,
				MaxIdrInterval: 2000,
			},
			Audio: avdata.ProbeAudio{
				Bitrate: 10,
				Codec:   "test_audio_codec",
			},
		},
	}
	meta, _ := RenditionMetadata(muxGenTestEncoderConfig, probeData)
	assert.Equal(4, len(meta))

	//Test Rendition metadata for source should be all defaults
	sourceRendition := meta[0]
	assert.Equal("chunked", sourceRendition.GroupID)
	assert.Equal(100, sourceRendition.VideoHeight)
	assert.Equal("test_codec,test_audio_codec", sourceRendition.Codecs)
	assert.Equal(200, sourceRendition.VideoWidth)
	assert.Equal(1060, sourceRendition.MuxRate)
	assert.Equal(int64(2000), sourceRendition.MaxIdrInterval)

	//Test all the rendition metadata generated for the passed in Test Encoder Configuration
	rendition := meta[1]
	assert.Equal("720p60", rendition.GroupID)
	assert.Equal(720, rendition.VideoHeight)
	assert.Equal("avc1.77.31,mp4a.40.2", rendition.Codecs)
	assert.Equal(1440, rendition.VideoWidth)
	assert.Equal(1680010, rendition.MuxRate)

	rendition = meta[2]
	assert.Equal("160p30", rendition.GroupID)
	assert.Equal(160, rendition.VideoHeight)
	assert.Equal("avc1.77.31,mp4a.40.2", rendition.Codecs)
	assert.Equal(320, rendition.VideoWidth)
	assert.Equal(230000, rendition.MuxRate)

	//Test all the rendition metadata for audio only group
	audioOnlyRendition := meta[3]
	assert.Equal("audio_only", audioOnlyRendition.GroupID)
	assert.Equal(0, audioOnlyRendition.VideoHeight)
	assert.Equal("test_audio_codec", audioOnlyRendition.Codecs)
	assert.Equal(0, audioOnlyRendition.VideoWidth)
	assert.Equal(10, audioOnlyRendition.MuxRate)

}

func TestVideoWidthGeneration(t *testing.T) {
	assert := assert.New(t)
	var probeData = avdata.ProbeResult{
		ProbeAV: avdata.ProbeAV{
			Video: avdata.ProbeVideo{
				Bitrate: 1000,
				Codec:   "test_codec",
				Profile: "test_profile",
				Height:  9,
				Width:   16,
			},
			Audio: avdata.ProbeAudio{
				Bitrate: 10,
				Codec:   "test_audio_codec",
			},
		},
	}

	var testEncoderConfig = &EncoderConfig{
		Transcodes: []TranscodeQualitySettings{
			TranscodeQualitySettings{
				Label:       "high",
				Bitrate:     1600000,
				DisplayName: "720p60_3100k",
				MaxFps:      60,
				Height:      480,
				CodecString: "avc1.77.31,mp4a.40.2",
			},
		},
	}

	meta, _ := RenditionMetadata(testEncoderConfig, probeData)
	assert.Equal(3, len(meta))

	//Test the video width generated , the mask should be applied to generate the correct width
	rendition := meta[1]
	assert.Equal(852, rendition.VideoWidth)
}
