package main

import (
	"context"
	"log"
	"os"
	"strconv"
	"time"

	"github.com/pkg/errors"
	uuid "github.com/satori/go.uuid"

	"code.justin.tv/video/gotranscoder/pkg/config"
	"code.justin.tv/video/gotranscoder/pkg/origin"
	"code.justin.tv/video/gotranscoder/pkg/tenfoot"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/protocols/hlsext"
)

type originSettings struct {
	endpoint         string
	originEnabled    bool
	channel          string
	transcodeProfile string
	resourcePath     string
	variants         []string
	usherStreamId    int
	fileFormat       hlsext.ExtendedPlaylist_FileFormat
	sessionId        string
}

func initOriginSession(endpoint string, channelID uint64, variants []string) (*originctl.Session, func(), error) {
	originClient := origin.New(origin.Config{
		Endpoint: endpoint,
	})
	session, exitCallback, err := originClient.StartSession(
		context.Background(), strconv.FormatUint(channelID, 10), variants)
	if err != nil {
		return nil, nil, err
	}
	return session, exitCallback, nil
}

func initOriginFoot(cfg *config.Values, sh *SegmentHandler, session *originctl.Session, runtimeStart time.Time, settings *originSettings) error {
	// Initialize and start the Tenfoot eventbus client.
	// Only init tenfoot if we are not in shadow mode
	var err error
	if cfg.Tenfoot.Enabled && !settings.originEnabled {
		sh.tenfootGlue, err = initTenfoot(cfg, runtimeStart, settings)
		if err != nil {
			return errors.Wrap(err, "error while initializing Tenfoot")
		}
	}
	if session == nil {
		return errors.New("session not established with origin")
	}

	metadata, err := newTenfootMetadata(cfg, runtimeStart, settings)
	if err != nil {
		return errors.Wrap(err, "unable to build tenfoot metadata")
	}
	// If cfg.Tenfoot.Enabled, then sh.tenfootGlue will be a real connection to
	// tenfoot, which we should wrap and use as the base for originfoot. If not,
	// then it will be a noop client, so wrapping it is harmless. So, we wrap it
	// regardless.
	sh.tenfootGlue = tenfoot.NewOriginFoot(sh.tenfootGlue, origin.Config{Endpoint: settings.endpoint}, session, metadata)
	return nil
}

func initTenfoot(cfg *config.Values, runtimeStart time.Time, settings *originSettings) (tenfoot.Glue, error) {
	tenfootSendTimeout, err := time.ParseDuration(cfg.Tenfoot.SendTimeout)
	if err != nil {
		return nil, errors.Wrap(err, "failed to parse send timeout")
	}

	tenfootShutdownTimeout, err := time.ParseDuration(cfg.Tenfoot.ShutdownTimeout)
	if err != nil {
		return nil, errors.Wrap(err, "failed to parse shutdown timeout")
	}

	tenfootSettings := &tenfoot.Settings{
		SendTimeout:     tenfootSendTimeout,
		ShutdownTimeout: tenfootShutdownTimeout,
		SockBasePath:    cfg.Tenfoot.SockBasePath,
		LogLevel:        cfg.Tenfoot.LogLevel,
	}

	tenfootMetadata, err := newTenfootMetadata(cfg, runtimeStart, settings)
	if err != nil {
		return nil, errors.Wrap(err, "failed to build tenfoot metadata")
	}

	ctx, cancel := context.WithCancel(context.Background())

	glue, err := tenfoot.New(ctx, tenfootSettings, tenfootMetadata)
	if err != nil {
		cancel()
		return nil, errors.Wrap(err, "failed to initialize Tenfoot client")
	}

	onExit.Add(func() {
		cancel()
		if err := glue.Wait(); err != nil {
			log.Printf("failed to gracefully stop Tenfoot client: %v", err)
		}
	})
	return glue, nil
}

func newTenfootMetadata(cfg *config.Values, runtimeStart time.Time, settings *originSettings) (*tenfoot.TranscodeMetadata, error) {
	tenfootSessionID, err := uuid.FromString(settings.sessionId)
	if err != nil {
		return nil, errors.Wrap(err, "failed to parse session ID")
	}

	host, err := os.Hostname()
	if err != nil {
		return nil, errors.Wrap(err, "failed to determine hostname")
	}

	targetDurationSecs, err := strconv.Atoi(cfg.Segment.TargetDuration)
	if err != nil {
		return nil, errors.Wrap(err, "failed to convert target duration from string")
	}

	tenfootMetadata := &tenfoot.TranscodeMetadata{
		Channel:               settings.channel,
		TranscodeProfile:      settings.transcodeProfile,
		StartTimestamp:        runtimeStart,
		TargetSegmentDuration: time.Duration(targetDurationSecs) * time.Second,
		Qualities:             settings.variants,
		UsherStreamID:         int64(settings.usherStreamId),
		SessionID:             tenfootSessionID,
		ResourcePath:          settings.resourcePath,
		OriginHostname:        host,
		FileFormat:            settings.fileFormat,
	}
	return tenfootMetadata, nil
}
