package main

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"testing"

	"github.com/stretchr/testify/assert"
)

var (
	testQualities = []string{"tmp_0", "tmp_1", "tmp_2", "tmp_3", "audio_only"}
)

func setupFakeTranscodePath(dirCount int, fileCount int) string {
	dir, err := ioutil.TempDir("", "output")
	if err != nil {
		log.Fatal(err)
	}

	for i := 0; i <= dirCount; i++ {
		tmp := filepath.Join(dir, fmt.Sprintf("tmp_%d", i))
		os.MkdirAll(tmp, os.ModePerm)

		// 4 files in dir
		for j := 0; j <= fileCount; j++ {
			tmpFile := filepath.Join(tmp, fmt.Sprintf("file_%d", j))
			if err := ioutil.WriteFile(tmpFile, []byte("temporary"), os.ModePerm); err != nil {
				log.Fatal(err)
			}
		}
	}

	return dir
}

func setupFakeTranscodePathWithoutAudio(dirCount, fileCount int) string {
	dir, err := ioutil.TempDir("", "output")
	if err != nil {
		log.Fatal(err)
	}

	for i := 0; i <= dirCount; i++ {
		tmp := filepath.Join(dir, fmt.Sprintf("tmp_%d", i))
		os.MkdirAll(tmp, os.ModePerm)
		//dont write audio_only files
		if testQualities[i] != "audio_only" {
			// 4 files in dir
			for j := 0; j <= fileCount; j++ {
				tmpFile := filepath.Join(tmp, fmt.Sprintf("file_%d", j))
				if err := ioutil.WriteFile(tmpFile, []byte("temporary"), os.ModePerm); err != nil {
					log.Fatal(err)
				}
			}
		}
	}

	return dir
}

// Full Directory
func TestTranscodeAvailableCheck(t *testing.T) {
	assert := assert.New(t)

	dir := setupFakeTranscodePath(4, 6)
	defer os.RemoveAll(dir) // clean up

	available := TranscodeAvailable(dir, testQualities)
	assert.True(available)
}

// Test transcode marked as ready when
func TestTranscodeNotAvailableCheck(t *testing.T) {
	assert := assert.New(t)

	dir := setupFakeTranscodePath(2, 6)
	defer os.RemoveAll(dir) // clean up

	available := TranscodeAvailable(dir, testQualities)
	assert.False(available)
}

// Test Full path no files
func TestEmptyTranscodeDirectory(t *testing.T) {
	assert := assert.New(t)

	dir := setupFakeTranscodePath(6, 0)
	defer os.RemoveAll(dir) // clean up

	available := TranscodeAvailable(dir, testQualities)
	assert.False(available)
}

// Test invalid transcoder path
func TestInvalidTranscodePath(t *testing.T) {
	assert := assert.New(t)

	available := TranscodeAvailable("/tmp", testQualities)
	assert.False(available)
}

//Test missing audio_only
func TestMissingAudioOnly(t *testing.T) {
	assert := assert.New(t)

	dir := setupFakeTranscodePathWithoutAudio(4, 6)
	defer os.RemoveAll(dir) // clean up
	available := TranscodeAvailable(dir, testQualities)

	assert.True(available)
}
