package main

import (
	"container/list"
	"fmt"
	"io/ioutil"
	"log"
	"math"
	"os"
	"path/filepath"
	"reflect"
	"strings"
	"sync"
	"time"

	"github.com/cenkalti/backoff"
	"github.com/golang/protobuf/proto"
	"github.com/golang/protobuf/ptypes/duration"
	"github.com/pkg/errors"

	"encoding/json"

	afg "code.justin.tv/ads/fring/grpc"
	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/tenfoot"
	"code.justin.tv/video/gotranscoder/pkg/usher"
	"code.justin.tv/video/protocols/common"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
)

// IdrMaxHistory is the number of IDR elements to keep track of.
// Use to calculate the historic IDR average for the broadcaster dashboard
const IdrMaxHistory = 30

// SegmentHistory is the list of seen segments and ad insertion
// moments per label
type SegmentHistory struct {
	sync.Mutex
	Segments      []avdata.Segment
	AdBreaks      []*hlsext.AdBreakRequest
	StreamOffsets map[string]*m3u8.StreamOffset
}

// InitSegments tracks the initialization segments required for Fmp4 packaged streams
func newSegmentHistory() *SegmentHistory {
	return &SegmentHistory{
		Segments:      make([]avdata.Segment, 0, 50),
		AdBreaks:      make([]*hlsext.AdBreakRequest, 0),
		StreamOffsets: make(map[string]*m3u8.StreamOffset),
	}
}

// InitSegments tracks the initialization segments required for Fmp4 packaged streams
type InitSegments struct {
	fmp4InitSegments map[string]*avdata.Fmp4InitSegment
	sync.Mutex
}

func newInitSegments() *InitSegments {
	return &InitSegments{
		fmp4InitSegments: make(map[string]*avdata.Fmp4InitSegment),
	}
}

// Get returns init segment for a label
func (is *InitSegments) Get(label string) (*avdata.Fmp4InitSegment, error) {
	is.Lock()
	defer is.Unlock()

	if _, ok := is.fmp4InitSegments[label]; !ok {
		return nil, errors.Errorf("missing init segment for label: %s", label)
	}

	return is.fmp4InitSegments[label], nil
}

// Set stores init segment for a label
func (is *InitSegments) Set(label string, seg avdata.Fmp4InitSegment) {
	is.Lock()
	defer is.Unlock()

	is.fmp4InitSegments[label] = &seg
}

// SegmentHandler Plugin interface to handle incoming segments
type SegmentHandler struct {
	// Keep track of metadata from all incoming segments and
	// AdBreakRequests. Should only be accessed under the historiesLock.
	historiesLock  sync.Mutex
	segmentHistory map[string]*SegmentHistory

	// Clients to the external world
	usher usher.Usher

	// References to the different renditions (one per quality level)
	// being produced
	streams map[string]*common.Stream

	// Segment Variables
	SegmentPath         string
	SegmentWindow       int
	TranscodeQualities  []string
	TargetDuration      int
	segmentHandlerStart time.Time
	FinalPlaylist       bool

	// Transcode tracking
	transcodeEntry *usher.HlsTranscode
	TranscodeID    int
	transcodeURI   string

	//// Broadcaster dashboard historicals
	// IDR
	idrHistory      *list.List // (IdrMaxHistory) elements queue.
	maxIdrDeltaSeen float64

	// Bitrate tracking
	maxKbpsSeen float64
	avgKbpsSeen float64
	kbpsSamples int64

	// FrameCount tracking
	maxFpsSeen float64
	avgFpsSeen float64
	fpsSamples int64

	// Ticker intervals
	transcodeUsherUpdateInterval time.Duration
	usherUpdateStreamInterval    time.Duration
	// Signaler to control ticker goroutines
	done chan struct{}
	wg   sync.WaitGroup

	tenfootGlue      tenfoot.Glue
	FileFormat       hlsext.ExtendedPlaylist_FileFormat
	fmp4InitSegments *InitSegments
}

// Initialize the segment history
// Initialized all segment information containers and starts the tickers
// in charge of updating stream related metadata.
func (sh *SegmentHandler) Initialize() {
	sh.segmentHistory = make(map[string]*SegmentHistory)
	sh.streams = make(map[string]*common.Stream)
	sh.idrHistory = list.New()
	sh.segmentHandlerStart = time.Now()
	sh.done = make(chan struct{})
	sh.tenfootGlue = tenfoot.Noop()

	segmentTrackingPlugin.TranscodeQualities = make([]string, 0, 10)
	sh.fmp4InitSegments = newInitSegments()

	exp := backoff.NewExponentialBackOff()
	exp.MaxElapsedTime = 30 * time.Second
	err := backoff.Retry(func() error {
		var attemptErr error
		sh.transcodeEntry, attemptErr = sh.usher.GetHlsTranscode(sh.TranscodeID)
		return attemptErr
	}, exp)
	if err != nil {
		log.Fatal("Failed to get transcode info entry from usher", err)
	}
	// Ticker to send usher updates with transcode information
	sh.wg.Add(1)
	go func() {
		defer sh.wg.Done()
		ticker := time.NewTicker(sh.transcodeUsherUpdateInterval)
		for {
			select {
			case <-ticker.C:
				statsd.Inc(constUsherTranscodeUpdate, 1, 1)

				exp := backoff.NewExponentialBackOff()
				exp.MaxElapsedTime = 2 * time.Second

				operation := func() error {
					return sh.UpdateUsherTranscode(ProbeResult)
				}

				err := backoff.Retry(operation, exp)
				if err != nil {
					log.Printf("[TICKER] error updating usher entry for transcodes %s", err)
					statsd.Inc(constUsherTranscodeUpdErr, 1, 1.0)
				}

			case <-sh.done:
				ticker.Stop()
				return
			}
		}
	}()

	// Ticker to send usher stream codec info updates for the health dashboard
	sh.wg.Add(1)
	go func() {
		defer sh.wg.Done()
		ticker := time.NewTicker(sh.usherUpdateStreamInterval)

		for {
			select {
			case <-ticker.C:
				// First segment may be delayed in the encoder
				// prevent race condition by pushing updates only after segment exists.
				sh.historiesLock.Lock()
				chunkedHistory, ok := sh.segmentHistory["chunked"]
				sh.historiesLock.Unlock()

				if ok {
					lastSegment := avdata.Segment{}

					chunkedHistory.Lock()
					lastItemIndex := len(chunkedHistory.Segments) - 1
					if lastItemIndex >= 0 {
						lastSegment = chunkedHistory.Segments[lastItemIndex]
					}
					chunkedHistory.Unlock()

					if !reflect.DeepEqual(lastSegment, avdata.Segment{}) {
						statsd.Inc(constUsherChanPropsUpdate, 1, 1)
						sh.UpdateStreamInfo(lastSegment)
					}
				}
			case <-sh.done:
				ticker.Stop()
				return
			}
		}
	}()
}

// Shuts down the SegmentHandler's tickers. Used to keep tests from leaking.
func (sh *SegmentHandler) close() {
	close(sh.done)
}

//UpdateUsherTranscode sends a transcode update to usher ever X seconds
func (sh *SegmentHandler) UpdateUsherTranscode(probeRes avdata.ProbeResult) error {
	meta, err := RenditionMetadataFromTranscodes(encoderConfig.Transcodes, probeRes)
	if err != nil {
		log.Println("Error in generating rendition metadata", err)
	}

	jsonMeta, err := json.Marshal(meta)
	if err != nil {
		log.Println("Error in marshalling rendition metadata", err)
	}

	sh.transcodeEntry.Status = "active"
	sh.transcodeEntry.Formats = strings.Join(sh.TranscodeQualities, ",")
	sh.transcodeEntry.RenditionMetadata = string(jsonMeta)
	sh.transcodeEntry.Version = int(*transcodeVersion)
	sh.transcodeEntry.TranscodeProfile = *transcodeProfile
	sh.transcodeEntry.URI = sh.transcodeURI

	log.Printf("[Send Usher update]: %+v", sh.transcodeEntry)
	return sh.usher.UpdateHlsTranscode(sh.TranscodeID, sh.transcodeEntry)
}

// Name describes this plugin, as as processor tag
func (sh *SegmentHandler) Name() string {
	return "IncomingSegmentHandler"
}

// Process an incoming segment or AdBreakRequest
func (sh *SegmentHandler) Process(val interface{}, timeoutCallback <-chan struct{}) error {
	switch typedVal := val.(type) {
	case avdata.Segment:
		return sh.processVideoSegment(typedVal)
	case *hlsext.AdBreakRequest:
		return sh.processAdBreakRequest(typedVal)
	case avdata.Fmp4InitSegment:
		return sh.processInitSegment(typedVal)
	default:
		log.Printf("SegmentHandler.Process received unexpected type: %T", val)
		return nil
	}
}

func (sh *SegmentHandler) processInitSegment(seg avdata.Fmp4InitSegment) error {
	sh.fmp4InitSegments.Set(seg.Label, seg)
	return nil
}

func (sh *SegmentHandler) processVideoSegment(seg avdata.Segment) error {
	// Keep track of the segment
	sh.Log(seg)

	sh.historiesLock.Lock()
	defer sh.historiesLock.Unlock()

	history, ok := sh.segmentHistory[seg.Label]
	if !ok {
		// First time
		history = newSegmentHistory()
		sh.segmentHistory[seg.Label] = history

		// initialize all playlists
		history.StreamOffsets[constLivePlaylist] = &m3u8.StreamOffset{}
		history.StreamOffsets[constHistoryPlaylist] = &m3u8.StreamOffset{}
		history.StreamOffsets[constWeaverPlaylist] = &m3u8.StreamOffset{}

		sh.TranscodeQualities = append(sh.TranscodeQualities, seg.Label)

		stream, err := newStream(*channel, seg.Label, usherTranscode.Destination)
		if err != nil {
			log.Printf("failed to make stream: %s", err)
			return err
		}
		sh.streams[seg.Label] = stream
	}

	sh.wg.Add(1)
	go func() {
		defer sh.wg.Done()

		history.Lock()
		defer history.Unlock()

		history.Segments = append(history.Segments, seg)
		if err := sh.tenfootGlue.PostSegment(&seg); err != nil {
			log.Printf("failed to post segment to Tenfoot: %v", err)
			statsd.Inc(constTenfootSegmentFailure, 1, 1.0)
		}
		wErr := sh.writeLivePlaylist(constTranscodeLivePlaylistWindowSize, seg.Label, fmt.Sprintf("%s/%s", sh.SegmentPath, seg.Label), constLivePlaylist, history)
		if wErr != nil {
			log.Println("[PLAYLIST] Failed to write live playlist", wErr)
		}
		wErr = sh.writeLivePlaylist(constTranscodeHistoryPlaylistWindowSize, seg.Label, fmt.Sprintf("%s/%s", sh.SegmentPath, seg.Label), constHistoryPlaylist, history)
		if wErr != nil {
			log.Println("[PLAYLIST] Failed to write live playlist", wErr)
		}
		wErr = sh.writeWeaverPlaylist(constTranscodeWeaverPlaylistWindowSize, seg.Label, fmt.Sprintf("%s/%s", sh.SegmentPath, seg.Label), history)
		if wErr != nil {
			log.Println("[PLAYLIST] Failed to write live playlist", wErr)
		}
	}()

	return nil
}

func (sh *SegmentHandler) processAdBreakRequest(req *hlsext.AdBreakRequest) error {
	// Logic will be:
	//  - Fill out Duration (maybe?)
	//  - For each label
	//    - Fill out StartSequenceNumber

	// Briefly grab the lock so we can take a snapshot of the current
	// value of sh.segmentHistory, copying it. Release the lock as soon
	// as possible so we don't block processSegment.
	sh.historiesLock.Lock()
	hists := make(map[string]*SegmentHistory)
	for label, h := range sh.segmentHistory {
		hists[label] = h
	}
	sh.historiesLock.Unlock()

	// Key the StartSequenceNumber field off of the "source" AKA "chunked" stream.
	sourceHist, ok := hists["chunked"]
	if !ok {
		// We haven't even got the first chunk of the broadcast! Just skip
		// this AdBreakRequest.
		return nil
	}
	sourceHist.Lock()
	// Insert the ad after the most recent source chunk's sequence number
	req.StartSequenceNumber = sourceHist.Segments[len(sourceHist.Segments)-1].SegmentNumber + 1
	sourceHist.Unlock()

	// Add the req to every quality level.
	for label, hist := range hists {
		hist.Lock()
		hist.AdBreaks = append(hist.AdBreaks, req)
		wErr := sh.writeWeaverPlaylist(constTranscodeWeaverPlaylistWindowSize, label, fmt.Sprintf("%s/%s", sh.SegmentPath, label), hist)
		hist.Unlock()
		if wErr != nil {
			log.Println("Failed to write weaver playlist", wErr)
		}
	}

	// Following set of lines replicate the legacy telegraph loop. This writes to transcoder process'
	// stdin on a pubsub message. Transcoder injects ad metadata to the video segements.
	// The player is able to trigger legacy ads when it encounters this ID3 tag

	//Decode ad duration from meta data
	adMetaData := new(afg.ChannelParams)
	if err := proto.Unmarshal(req.AdData, adMetaData); err != nil {
		log.Printf("[AD INSERTION] proto unmarshal err: %s, using default length of 30 Seconds", err)
		statsd.Inc(constLegacyAdDecodeFailure, 1, 1.0)
		if err = transcoder.WriteToStdin(fmt.Sprintf(legacyAdBreakMetadata+"\n", 30)); err != nil {
			log.Printf("[AD INSERTION] Failed to write to transcoder's stdin with: %s", err)
			statsd.Inc(constLegacyAdInsertionFailure, 1, 1.0)
		}
		return nil
	}
	log.Println("[AD INSERTION]", adMetaData)
	if err := transcoder.WriteToStdin(fmt.Sprintf(legacyAdBreakMetadata+"\n", adMetaData.BreakLength)); err != nil {
		log.Printf("[AD INSERTION] Failed to write to transcoder's stdin with: %s", err)
		statsd.Inc(constLegacyAdInsertionFailure, 1, 1.0)
	}
	return nil
}

// Log an incoming chunk
func (sh *SegmentHandler) Log(segment avdata.Segment) {
	log.Printf("[Segment]: %+v \n", segment)
}

// ForcePlaylistFlush Force writes all playlists
func (sh *SegmentHandler) ForcePlaylistFlush() {
	log.Printf("[Flushing playlists]")

	// make copy of current sh.segmentHistory
	sh.historiesLock.Lock()
	hists := make(map[string]*SegmentHistory)
	for label, h := range sh.segmentHistory {
		hists[label] = h
	}
	sh.historiesLock.Unlock()

	for k, hist := range hists {
		hist.Lock()
		wErr := sh.writeLivePlaylist(constTranscodeLivePlaylistWindowSize, k, fmt.Sprintf("%s/%s", sh.SegmentPath, k), constLivePlaylist, hist)
		if wErr != nil {
			log.Println("Failed to write live playlist", wErr)
		}
		wErr = sh.writeLivePlaylist(constTranscodeHistoryPlaylistWindowSize, k, fmt.Sprintf("%s/%s", sh.SegmentPath, k), constHistoryPlaylist, hist)
		if wErr != nil {
			log.Println("Failed to write live playlist", wErr)
		}
		wErr = sh.writeWeaverPlaylist(constTranscodeWeaverPlaylistWindowSize, k, fmt.Sprintf("%s/%s", sh.SegmentPath, k), hist)
		if wErr != nil {
			log.Println("Failed to write weaver playlist", wErr)
		}
		hist.Unlock()
	}
}

// Should be run with sh.segmentHistory[label].Lock()
func (sh *SegmentHandler) weaverPlaylist(window int, label string, history *SegmentHistory) (*hlsext.ExtendedPlaylist, error) {
	m3u8Playlist, err := sh.m3u8(window, label, constWeaverPlaylist, history)
	if err != nil {
		return nil, err
	}

	hlsPlaylist, err := m3u8Playlist.ToHLSProto(time.Now())
	if err != nil {
		return nil, err
	}

	pl := &hlsext.ExtendedPlaylist{
		Version:  0,
		Stream:   sh.streams[label],
		Playlist: hlsPlaylist,
		Ads:      sh.windowedAdBreakReqs(window, label, history),
		Bitrate:  sh.weaverBitrate(label),
	}
	if sh.FileFormat == hlsext.ExtendedPlaylist_MP4 {
		pl.FileFormat = hlsext.ExtendedPlaylist_MP4
		initSegName, err := sh.getInitSegmentName(label)
		if err != nil {
			return nil, err
		}
		pl.InitSegmentUri = initSegName
	}

	return pl, nil
}

func (sh *SegmentHandler) weaverBitrate(label string) int32 {
	// If 'label' refers to a transcode, find the transcode config whose
	// label matches. Return the bitrate we're targeting with the
	// transcode.
	for _, transcode := range encoderConfig.Transcodes {
		if transcode.Label == label {
			var videoBitrate int32
			if transcode.Muxrate > 0 {
				videoBitrate = transcode.Muxrate - (transcode.AudioBitrate * transcode.AudioChannels)
			} else {
				videoBitrate = transcode.Bitrate // The transcode.Bitrate is already in bps
			}
			return videoBitrate
		}
	}
	// If not, this is the source stream. Return the observed bitrate
	return int32(1024 * sh.avgKbpsSeen) // 1024 to convert kbps to bps
}

// Compute the list of futureSegments attached to the last semgent in our history
// Please note that history should be locked before this function is called (history.Lock())
func (sh *SegmentHandler) twitchTranscoderFutureSegments(history *SegmentHistory) []*hls.Segment {
	out := []*hls.Segment{}

	if history != nil && len(history.Segments) > 0 {
		lastSegment := history.Segments[len(history.Segments)-1]

		for idx, seg := range lastSegment.FutureSegments {
			// duration is just an estimate, use any of the other chunks' durations as substitute
			segmentNearestIntDuration := (lastSegment.Duration / 1000.) + 0.5
			hlsSegment := &hls.Segment{
				Uri:            seg,
				SequenceNumber: lastSegment.SegmentNumber + int64(idx+1),
				// Divide duration by 1000 to get from ms to seconds
				Duration:      &duration.Duration{int64(segmentNearestIntDuration), 0},
				Discontinuity: false,
			}
			out = append(out, hlsSegment)
		}
	}
	return out
}

// Compute the list of AdBreakRequests that are within the last
// 'window' segments for the label playlist. Should be run with
// sh.segmentHistory[label].Lock()
func (sh *SegmentHandler) windowedAdBreakReqs(window int, label string, history *SegmentHistory) []*hlsext.AdBreakRequest {
	// Compute the sequence number of the oldest segment, up to 'window' chunks in the past.
	var firstSeqNum int64
	totalChunks := len(history.Segments)
	if window > totalChunks {
		firstSeqNum = 0
	} else {
		firstSeqNum = int64(totalChunks - window - 1)
	}

	// Filter out AdBreakRequests that would be inserted before the oldest segment
	var out []*hlsext.AdBreakRequest
	for _, req := range history.AdBreaks {
		if req.StartSequenceNumber < firstSeqNum {
			continue
		}
		out = append(out, req)
	}
	return out
}

func (sh *SegmentHandler) m3u8(window int, label string, playListType string, history *SegmentHistory) (m3u8.Playlist, error) {
	// Playlist Boundaries
	totalChunks := len(history.Segments)
	items := window
	if totalChunks <= items {
		items = totalChunks
	}
	startingChunk := totalChunks - items
	if startingChunk < 0 {
		startingChunk = 0
	}

	// update the elapse time
	streamOffset, ok := history.StreamOffsets[playListType]
	if !ok {
		log.Printf("[PLAYLIST] - Failed to find playlistType(%s), adding it now", playListType)
		history.StreamOffsets[playListType] = &m3u8.StreamOffset{}
		streamOffset = history.StreamOffsets[playListType]
	}

	m3u8.Elapsed(streamOffset, &history.Segments, startingChunk)

	// Playlist target duration is a single digit second, rounded up from max target duration
	// e.g.: 5 for segments of size 4100
	var targetDuration int
	if sh.TargetDuration%1000 == 0 {
		targetDuration = sh.TargetDuration / 1000
	} else {
		targetDuration = sh.TargetDuration/1000 + 1
	}

	playlist := m3u8.Playlist{
		Chunks:         make([]m3u8.Chunk, 0, window),
		IsFinal:        sh.FinalPlaylist,
		WindowSize:     window,
		MediaSequence:  int(math.Max(float64(totalChunks-window), 0.0)),
		TargetDuration: targetDuration,
		StreamDuration: time.Since(sh.segmentHandlerStart).Seconds(),
		StreamOffset:   streamOffset.TimeElapsed,
	}
	if sh.FileFormat == hlsext.ExtendedPlaylist_MP4 {
		initSegName, err := sh.getInitSegmentName(label)
		if err != nil {
			return m3u8.Playlist{}, err
		}
		playlist.InitSegmentUri = initSegName
	}

	for i := startingChunk; i < totalChunks; i++ {
		chunk := m3u8.Chunk{
			SequenceNumber: history.Segments[i].SegmentNumber,
			URL:            history.Segments[i].SegmentName,
			Duration:       history.Segments[i].Duration,
		}

		if history.Segments[i].ProgramDateTime > 0 {
			programDateTime := time.Unix(0, history.Segments[i].ProgramDateTime*int64(time.Millisecond))
			chunk.ProgramDateTime = &programDateTime
		}

		playlist.Chunks = append(playlist.Chunks, chunk)
	}
	return playlist, nil
}

// writeLivePlaylist writes to disk an M3u8 playlist with the last [window] transcoded segments
// Please note that history should be locked before this function is called (history.Lock())
func (sh *SegmentHandler) writeLivePlaylist(window int, label string, path string, name string, history *SegmentHistory) error {
	if err := os.MkdirAll(path, 755); err != nil {
		return errors.Wrapf(err, "failed to create directory %s", path)
	}

	playlist, err := sh.m3u8(window, label, name, history)
	if err != nil {
		return err
	}

	playlistPath := fmt.Sprintf("%s/%s-%s.m3u8", path, constPlaylistPrefix, name)
	pyPlaylistPath := fmt.Sprintf("%s/py-%s-%s.m3u8", path, constPlaylistPrefix, name)

	if sh.FileFormat == hlsext.ExtendedPlaylist_TS {
		err := sh.atomicWriteFile(playlistPath, playlist.GenerateV3())
		if err != nil {
			return errors.Wrapf(err, "failed to write playlist %s", playlistPath)
		}

		err = sh.atomicWriteFile(pyPlaylistPath, playlist.GenerateV3())
		if err != nil {
			return errors.Wrapf(err, "failed to write playlist %s", pyPlaylistPath)
		}
	} else {

		err := sh.atomicWriteFile(playlistPath, playlist.GenerateV6())
		if err != nil {
			return errors.Wrapf(err, "failed to write playlist %s", playlistPath)
		}

		err = sh.atomicWriteFile(pyPlaylistPath, playlist.GenerateV6())
		if err != nil {
			return errors.Wrapf(err, "failed to write playlist %s", pyPlaylistPath)
		}
	}

	// @KK: TODO: Right now, `windowedAdBreakReqs` will be called twice for each new playlist: once here and once in
	// `weaverPlaylist`.  We could/should rearrange this so that a single `hlsext.ExtendedPlaylist` is generated.
	adBreaks := sh.windowedAdBreakReqs(window, label, history)
	futureSegments := sh.twitchTranscoderFutureSegments(history)

	initSegName, err := sh.getInitSegmentName(label)
	if err != nil {
		return errors.Wrapf(err, "Error getting init segment name for label: %s", label)
	}

	if err := sh.tenfootGlue.PostPlaylist(playlist, label, playlistPath, name, adBreaks, futureSegments, initSegName); err != nil {
		statsd.Inc(constTenfootPlaylistFailure, 1, 1.0)
		return errors.Wrap(err, "[PLAYLIST] - Failed to post playlist to Tenfoot")
	}

	return nil
}

// writeWeaverPlaylist handles writing a protocol buffers formatted video playlist for Weaver - the playlist service
func (sh *SegmentHandler) writeWeaverPlaylist(window int, label string, path string, history *SegmentHistory) error {
	if err := os.MkdirAll(path, 755); err != nil {
		return errors.Wrapf(err, "[WEAVER PLAYLIST] - Failed to ensure that the %s directory exists:", path)
	}

	playlist, err := sh.weaverPlaylist(window, label, history)
	if err != nil {
		return errors.Wrap(err, "[WEAVER PLAYLIST] - Failed to generate Weaver playlist")
	}

	bytes, err := proto.Marshal(playlist)
	if err != nil {
		return errors.Wrap(err, "[WEAVER PLAYLIST] - Failed to marshal Weaver playlist")
	}

	if err := sh.atomicWriteFile(filepath.Join(path, "playlist.pb"), bytes); err != nil {
		return errors.Wrap(err, "[WEAVER PLAYLIST] - Failed to write file to disk")
	}
	return nil
}

func (sh *SegmentHandler) atomicWriteFile(path string, content []byte) error {
	tmpPath := fmt.Sprintf("%s.tmp", path)
	err := ioutil.WriteFile(tmpPath, content, 0644)
	if err != nil {
		return errors.Wrapf(err, "Error writing %s", tmpPath)
	}

	err = os.Rename(tmpPath, path)
	if err != nil {
		return errors.Wrapf(err, "Error renaming temp file %s", tmpPath)
	}

	return nil
}

// UpdateStreamInfo will take a segment of type CHUNKED(source),
// and extract IDR, bitrate and FPS parameters.
// This will update usher with the average bitrate and IDR to power
// the stream quaility tag in the broadcaster dashboard.
//
// To avoid hammering usher. this should only post the data every minute.
func (sh *SegmentHandler) UpdateStreamInfo(segment avdata.Segment) {
	if segment.Label != "chunked" {
		log.Println("Segment not of type Chunked - Nothing  to report")
	}

	idrMaxDelta, idrAlignedPerc, err := sh.IdrParameters(segment)
	if err != nil {
		log.Printf("Error retrieving IDR calculation for segment %+v\n%s\n", segment, err)
	}

	kbpsMax, kbpsAvg, err := sh.BitrateParamenters(segment)
	if err != nil {
		log.Printf("Error retrieving KBPS calculation for segment %+v\n%s\n", segment, err)
	}
	//We update the ProbeResult with the latest bitrate value for source , so that we can send update
	ProbeResult.ProbeAV.Video.Bitrate = int64(kbpsAvg * 1000.0)
	bitrateUpdated = true

	_, fpsMax, _, err := sh.FramerateParamenters(segment)
	if err != nil {
		log.Printf("Error retrieving FPS calculation for segment %+v\n%s\n", segment, err)
	}

	streamUpdate := usher.StreamCodec{
		Channel:              *channel,
		ChannelID:            *channelID,
		MaxIdrDelta:          idrMaxDelta,
		IdrAlignedPercentage: idrAlignedPerc,
		AverageFps:           fpsMax,
		MaxKbps:              kbpsMax,
		AverageKbps:          kbpsAvg,
	}

	err = sh.usher.UpdateStreamProperties(channelPropertyID, &streamUpdate)
	if err != nil {
		log.Printf("Error pushing codec info to usher %s\n", err)
	}
}

// IdrParameters returns ths IDR calculation of the segment history.
// This should only be done for SOURCE type segments
func (sh *SegmentHandler) IdrParameters(segment avdata.Segment) (maxDelta float64, AlignedPercentage float64, err error) {
	if segment.Label != "chunked" {
		return 0, 0, errors.New("Idr parameters only available for source segements")
	}

	if segment.MaxIdrDelta > sh.maxIdrDeltaSeen {
		sh.maxIdrDeltaSeen = segment.MaxIdrDelta
	}

	if segment.RandomAccessPoint == true {
		sh.idrHistory.PushBack(1.0)
	} else {
		sh.idrHistory.PushBack(0.0)
	}

	sum := 0.0
	for e := sh.idrHistory.Front(); e != nil; e = e.Next() {
		sum += e.Value.(float64)
	}

	idrPercentage := sum / float64(sh.idrHistory.Len())

	if sh.idrHistory.Len() > IdrMaxHistory {
		sh.idrHistory.Remove(sh.idrHistory.Front())
	}

	return sh.maxIdrDeltaSeen, idrPercentage, nil
}

// BitrateParamenters calculates the bitrate history and populates the average bitrate for the stream
// this is used to show the quality tag in the broadcaster dashboard.
func (sh *SegmentHandler) BitrateParamenters(segment avdata.Segment) (maxKbps float64, avgKbps float64, err error) {
	if segment.Duration <= 0 {
		return 0, 0, errors.New("No segment duration present to calculate bitrate")
	}

	var kbps float64
	kbps = (float64(segment.SegmentSize) * 8) / float64(segment.Duration)

	if kbps > sh.maxKbpsSeen {
		sh.maxKbpsSeen = kbps
	}

	sh.kbpsSamples++
	sh.avgKbpsSeen = (sh.avgKbpsSeen*(float64(sh.kbpsSamples)-1) + kbps) / float64(sh.kbpsSamples)

	return sh.maxKbpsSeen, sh.avgKbpsSeen, nil
}

// FramerateParamenters calculates the historical framerate of the stream
// used for the broadcaster dashboard.
func (sh *SegmentHandler) FramerateParamenters(segment avdata.Segment) (fps float64, maxFps float64, avgFps float64, err error) {

	fps = (float64(segment.FrameCount) * 1000) / float64(segment.Duration)

	if fps > sh.maxFpsSeen {
		sh.maxFpsSeen = fps
	}

	sh.fpsSamples++
	sh.avgFpsSeen = (sh.avgFpsSeen*(float64(sh.fpsSamples)-1) + fps) / float64(sh.fpsSamples)

	return fps, sh.maxFpsSeen, sh.avgFpsSeen, nil
}

func (sh *SegmentHandler) getInitSegmentName(label string) (string, error) {
	var initSeg *avdata.Fmp4InitSegment
	var err error
	if sh.FileFormat == hlsext.ExtendedPlaylist_MP4 {
		if initSeg, err = sh.fmp4InitSegments.Get(label); err != nil {
			return "", err
		}
		return initSeg.SegmentName, nil
	}
	// TS
	return "", nil
}
