package main

import (
	"fmt"
	"log"
	"os"
	"sync"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	streamlog "code.justin.tv/video/streamlog/pkg/wswriter"
)

var (
	onceTranscode     sync.Once
	onceSegDur        sync.Once
	onceVidCodec      sync.Once
	onceAudCodec      sync.Once
	onceTranscodeHost sync.Once
	onceIDR           sync.Once
)

// StreamlogHandler receives avdata for transmission to streamlog
type StreamlogHandler struct {
	frameDrop bool
}

// Initialize sets up the handler
func (s *StreamlogHandler) Initialize() {
	log.Println("[StreamlogPlugin] Streamlog logging plugin")
}

// Name returns the handler's name for logging purposes
func (s *StreamlogHandler) Name() string {
	return "StreamlogHandler"
}

// Process Logs Transcode Profile only once
// For Segment data, calls Segment Processor
// For Codec data, calls Codec Processor
func (s *StreamlogHandler) Process(data interface{}, timeout <-chan struct{}) error {
	onceTranscode.Do(func() {
		streamlog.Log(*channel, "transcode_profile", *transcodeProfile)
		streamlog.Log(*channel, "vod_enabled", *vodPusherEnabled)
	})

	onceTranscodeHost.Do(func() {
		hostname, err := os.Hostname()
		if err != nil {
			hostname = "undetected"
		}
		streamlog.Log(*channel, "transcode_host", hostname)
	})

	var err error
	switch d := data.(type) {
	case avdata.Segment:
		_, err = s.ProcessSegment(&d)
	case avdata.Codec:
		_, err = s.ProcessCodec(&d)
	case avdata.StreamLog:
		_, err = s.processSLData(&d)
	default:
		log.Printf("Streamlog Handler asked to process unknown type: %T\n", d)
		statsd.Inc("Streamlog.Unknown", 1, 1.0)
		return nil
	}
	return err
}

func (s *StreamlogHandler) processSLData(data *avdata.StreamLog) (map[string]interface{}, error) {
	// Report rounded up in Seconds to Statsd
	statsd.Inc(fmt.Sprintf("keyframe_interval.%d", int(data.SLData.SourceIDR/1000)), 1, 0.1)
	streamlogMap := make(map[string]interface{})
	onceIDR.Do(func() {
		streamlog.Log(*channel, "source_idr_interval", data.SLData.SourceIDR)
		streamlogMap["source_idr_interval"] = data.SLData.SourceIDR
	})

	return streamlogMap, nil
}

// ProcessSegment handles segment data parsed from TwitchTranscoder
func (s *StreamlogHandler) ProcessSegment(seg *avdata.Segment) (map[string]interface{}, error) {
	// Update Segment duration for first segment
	// Update Framerate data for every source segment
	if seg.Label == "chunked" {
		streamlogMap, err := s.logFramerate(seg)
		onceSegDur.Do(func() {
			streamlog.Log(*channel, "segment_duration", seg.Duration)
			streamlogMap["segment_duration"] = seg.Duration
		})
		return streamlogMap, err
	}
	return nil, nil
}

func (s *StreamlogHandler) logFramerate(seg *avdata.Segment) (map[string]interface{}, error) {
	sh := &SegmentHandler{}
	streamlogMap := make(map[string]interface{})
	fps, maxFps, _, _ := sh.FramerateParamenters(*seg)
	// If current Frame Rate is < threshold of the Max Frame rate
	// we will log a framedrop event and log the drop in fps
	// Log this stat only for Source
	var changed bool

	streamlog.Log(*channel, "frame_rate", fps)
	streamlogMap["frame_rate"] = fps

	if fps < cfg.Streamlog.FramedropThreshold*maxFps {
		// If frameDrop flag was False before, that means frameDrop has now STARTED
		// and the status has changed. Log framedrop_start event and drop in fps
		changed = !s.frameDrop
		s.frameDrop = true
		if changed {
			streamlog.Log(*channel, "framesdropped", s.frameDrop)
			streamlogMap["framesdropped"] = s.frameDrop
		}
	} else {
		// If frameDrop flag was True before, that means frameDrop has now STOPPED
		// and the status has changed. Log framedrop_stop event
		changed = s.frameDrop
		s.frameDrop = false
		if changed {
			streamlog.Log(*channel, "framesdropped", s.frameDrop)
			streamlogMap["framesdropped"] = s.frameDrop
		}
	}
	return streamlogMap, nil
}

// ProcessCodec handles codec data from TwitchTranscoder
func (s *StreamlogHandler) ProcessCodec(codec *avdata.Codec) (map[string]interface{}, error) {
	streamlogMap := make(map[string]interface{})
	if len(codec.VideoCodec) > 0 {
		onceVidCodec.Do(func() {
			streamlog.Log(*channel, "source_video_codec", codec.VideoCodec)
			streamlog.Log(*channel, "avc_level", codec.Level)
			streamlogMap["source_video_codec"] = codec.VideoCodec
			streamlogMap["avc_level"] = codec.Level
		})
	}
	if len(codec.AudioCodec) > 0 {
		onceAudCodec.Do(func() {
			streamlog.Log(*channel, "source_audio_codec", codec.AudioCodec)
			streamlogMap["source_audio_codec"] = codec.AudioCodec
		})
	}
	return streamlogMap, nil
}
