package main

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
)

func TestStreamlogHandlerProcessSegmentChunked(t *testing.T) {
	assert := assert.New(t)

	handler := StreamlogHandler{}
	handler.Initialize()

	streamlogMap, _ := handler.ProcessSegment(&avdata.Segment{
		Label:         "chunked",
		SegmentNumber: 1,
		FrameCount:    60,
		Duration:      1000,
	})

	expStreamlogMap := make(map[string]interface{})
	expStreamlogMap["segment_duration"] = 1000.0
	expStreamlogMap["frame_rate"] = 60.0

	assert.Equal(expStreamlogMap, streamlogMap)
}

func TestStreamlogHandlerProcessSegmentHigh(t *testing.T) {
	assert := assert.New(t)

	handler := StreamlogHandler{}
	handler.Initialize()

	var expStreamlogMap map[string]interface{}
	streamlogMap, _ := handler.ProcessSegment(&avdata.Segment{
		Label:         "high",
		SegmentNumber: 2,
	})

	assert.Equal(expStreamlogMap, streamlogMap)
}

func TestStreamlogHandlerProcessCodecVideo(t *testing.T) {
	assert := assert.New(t)

	handler := StreamlogHandler{}
	handler.Initialize()

	streamlogMap, _ := handler.ProcessCodec(&avdata.Codec{
		VideoCodec: "avc1.640033",
		Level:      51,
	})

	expStreamlogMap := make(map[string]interface{})
	expStreamlogMap["source_video_codec"] = "avc1.640033"
	expStreamlogMap["avc_level"] = 51

	assert.Equal(expStreamlogMap, streamlogMap)
}

func TestStreamlogHandlerProcessCodecAudio(t *testing.T) {
	assert := assert.New(t)

	handler := StreamlogHandler{}
	handler.Initialize()

	streamlogMap, _ := handler.ProcessCodec(&avdata.Codec{
		AudioCodec: "mp4a.40.2",
	})

	expStreamlogMap := make(map[string]interface{})
	expStreamlogMap["source_audio_codec"] = "mp4a.40.2"

	assert.Equal(expStreamlogMap, streamlogMap)
}

func TestStreamlogHandlerProcessSLData(t *testing.T) {
	assert := assert.New(t)

	handler := StreamlogHandler{}
	handler.Initialize()

	streamlogMap, _ := handler.processSLData(&avdata.StreamLog{
		SLData: avdata.StreamlogData{
			SourceIDR: 2400,
		},
	})

	expStreamlogMap := make(map[string]interface{})
	expStreamlogMap["source_idr_interval"] = int64(2400)

	assert.Equal(expStreamlogMap, streamlogMap)

	// Try logging again and make sure it doesn't log
	streamlogMap, _ = handler.processSLData(&avdata.StreamLog{
		SLData: avdata.StreamlogData{
			SourceIDR: 2400,
		},
	})
	assert.Equal(make(map[string]interface{}), streamlogMap)
}
