package main

import (
	"log"
	"math"
	"net"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"context"

	"github.com/aws/aws-sdk-go/aws"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/config"
	"code.justin.tv/video/gotranscoder/pkg/notify"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/vod"
)

func initVODPusher(cfg *config.Values, broadcastID int, encoder *EncoderConfig, hlsUrlBase string, notifier notify.Notifier, probeResult *avdata.ProbeResult) *vod.Pusher {
	targetSegDur, err := strconv.ParseFloat(cfg.Segment.TargetDuration, 10)
	if err != nil {
		log.Fatalf("Invalid target segment duration: %s\n", err)
	}

	// http.DefaultTransport with proxy if valid config
	transport := &http.Transport{
		Proxy: http.ProxyFromEnvironment,
		Dial: (&net.Dialer{
			Timeout:   30 * time.Second,
			KeepAlive: 30 * time.Second,
		}).Dial,
		TLSHandshakeTimeout: 3 * time.Second,
		MaxIdleConnsPerHost: 100,
		MaxIdleConns:        100,
	}
	if proxyURL, err := url.ParseRequestURI(cfg.VOD.Proxy); err == nil {
		transport.Proxy = http.ProxyURL(proxyURL)
	}

	// Want VOD chunks to be as close to the target object size as possible
	// Add .5 so we round to closest multiple (must be at least 1).
	chunkFactor := int(math.Max(1, cfg.VOD.TargetObjectDur/targetSegDur) + .5)

	s3Conn, err := vod.NewS3Connection(&aws.Config{
		Region:           aws.String("us-west-2"),
		HTTPClient:       &http.Client{Transport: transport},
		S3ForcePathStyle: aws.Bool(true),
	})

	if err != nil {
		log.Fatalf("[VOD]: Failed to initialize s3 Connection")
	}

	ps := vod.PusherSettings{
		ChannelName:         *channel,
		ChannelID:           *channelID,
		BroadcastID:         broadcastID,
		BroadcasterSoftware: GetBroadcaster(),
		VodKey:              transcodesFolder,
		TranscodePath:       transcodesBasePath,
		QueueSize:           cfg.VOD.QueueSize,
		NotifyQueueSize:     cfg.VOD.NotifyQueueSize,
		NumThumbnails:       cfg.VOD.NumThumbnails,
		ChunkFactor:         chunkFactor,
		PlaylistFileName:    constVODPlaylistName,
		VinylHost:           cfg.VOD.VinylHost,
		Usher:               usherCfg,
		Bucket:              cfg.VOD.S3Bucket,
		SoftDeleteVod:       *statelessMode,
		Conn:                s3Conn,
		MinVodDurationSeconds: cfg.VOD.MinVodDurationSeconds,
		HlsUrlBase:            hlsUrlBase,
		DisableVinyl:          *disableVinyl,
		Notifier:              notifier,
		MaxIdrInterval:        int(probeResult.ProbeAV.Video.MaxIdrInterval),
	}

	// Override S3 bucket if provided
	if *s3Bucket != "" {
		ps.Bucket = *s3Bucket
	}

	if *s3Prefix != "" {
		ps.Prefix = *s3Prefix
	}

	if *cdnUrl != "" {
		ps.VodDistributionUrl = *cdnUrl
	}

	log.Printf("[VOD] using config: %+v", ps)

	// Always add 'chunked' and 'audio_only' qualities
	qualities := map[string]vod.Quality{
		constTranscodeSource: {
			Label:              constTranscodeSource,
			DisplayName:        "Source",
			PlaylistPreference: 0,
			Fps:                float64(probeResult.ProbeAV.Video.Fps),
			Bitrate:            int(probeResult.ProbeAV.Video.Bitrate),
		},
		constTranscodeAudioOnly: {
			Label:              constTranscodeAudioOnly,
			DisplayName:        "Audio Only",
			PlaylistPreference: 5,
			Bitrate:            int(probeResult.ProbeAV.Audio.Bitrate),
		},
	}

	for _, t := range encoder.Transcodes {
		qualities[t.Label] = vod.Quality{
			Label:              t.Label,
			DisplayName:        t.DisplayName,
			PlaylistPreference: t.PlaylistPreference,
			Fps:                float64(t.MaxFps),
			Bitrate:            int(t.Bitrate),
		}
	}

	go LogVodConfiguration(&ps)

	p := vod.NewPusher(ps, qualities)
	onExit.Add(p.Quit) // Finalize VOD when we cleanly exit

	ctx, cancel := context.WithCancel(context.Background())
	checkPusherHealthAndLog(ctx, p) //This check is non-blocking
	onExit.Add(cancel)

	return p
}

//Checks vod pusher health and logs to elasticsearch
// 10 minutes after transcode start
func checkPusherHealthAndLog(ctx context.Context, pusher *vod.Pusher) {
	go func() {
		for {
			select {
			case <-ctx.Done():
				return
			case <-time.After(10 * time.Minute):
				if *vodPusherEnabled && !pusher.Health() {
					log.Println("[VOD]: vod pusher Unhealthy")
					statsd.Inc(constVodPusherUnheatlhy, 1, 1)
					LogVodPusherHealth(false)
					return
				}
				LogVodPusherHealth(true)
				return
			}
		}
	}()
}
