package m3u8

import (
	"bytes"
	"fmt"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
)

func (pl *Playlist) playlistHeader(version int) []byte {
	var hdr bytes.Buffer

	hdr.WriteString(HeaderFileHdr)
	switch version {
	case 3:
		hdr.WriteString(HeaderVersion3)
	case 6:
		hdr.WriteString(HeaderVersion6)
	default:
		hdr.WriteString(HeaderVersion3)

	}
	hdr.WriteString(fmt.Sprintf(HeaderTargetDuration, pl.TargetDuration))
	hdr.WriteString(fmt.Sprintf(HeaderEquivTdtg, time.Now().UTC().Format(DateTimeFormat)))

	switch pl.PlaylistType {
	case PlaylistTypeEvent:
		hdr.WriteString(HeaderPlaylistTypeEvent)
	case PlaylistTypeVod:
		hdr.WriteString(HeaderPlaylistTypeVod)
	}

	hdr.WriteString(fmt.Sprintf(MarkerMediaSequence, pl.MediaSequence))
	hdr.Write(pl.twitchPlaylistHeader())

	return hdr.Bytes()
}

func (pl *Playlist) elapsed() float64 {
	var chunkDuration float64

	for _, chunk := range pl.Chunks {
		chunkDuration += chunk.Duration
	}

	return pl.StreamDuration - (chunkDuration / 1000)
}

func (pl *Playlist) twitchPlaylistHeader() []byte {
	var hdr bytes.Buffer
	hdr.WriteString(fmt.Sprintf(TwitchElapsedSeconds, pl.StreamOffset))
	hdr.WriteString(fmt.Sprintf(TwitchTotalSeconds, pl.StreamDuration))
	return hdr.Bytes()
}

// GenerateV3 HLS playlist based on a container of video chunks.
func (pl *Playlist) GenerateV3() []byte {
	var playlist bytes.Buffer

	playlist.Write(pl.playlistHeader(3))
	playlist.Write(pl.writeSegments())

	if pl.IsFinal {
		playlist.WriteString(MarkerPlaylistEnd)
	}

	return playlist.Bytes()
}

func (pl *Playlist) writeSegments() []byte {
	var segments bytes.Buffer

	for _, chunk := range pl.Chunks {
		if chunk.URL == DiscontinuityURL {
			segments.WriteString(MarkerDiscontinuity)
			segments.WriteString(TwitchMarkerDiscontinuity)
		} else {
			segments.WriteString(fmt.Sprintf(MarkerSegment, (chunk.Duration / 1000), chunk.URL))
		}
	}

	return segments.Bytes()
}

// GenerateV6 generates a version 6 HLS playlist based on a container of video chunks
func (pl *Playlist) GenerateV6() []byte {
	var playlist bytes.Buffer

	playlist.Write(pl.playlistHeader(6))
	if pl.InitSegmentUri != "" {
		playlist.WriteString(fmt.Sprintf(HeaderMapUri, pl.InitSegmentUri))
	}
	playlist.Write(pl.writeSegments())

	if pl.IsFinal {
		playlist.WriteString(MarkerPlaylistEnd)
	}

	return playlist.Bytes()
}

// Elapsed sets the TimeElapsed field in streamOffset based on the segment list up to startingChunk
func Elapsed(streamOffset *StreamOffset, segments *[]avdata.Segment, startingChunk int) {
	for streamOffset.CurSegIdx < startingChunk {
		streamOffset.TimeElapsed += (*segments)[streamOffset.CurSegIdx].Duration / 1000
		streamOffset.CurSegIdx++
	}
}
