package m3u8

import (
	"math"
	"time"

	"github.com/golang/protobuf/ptypes"

	"code.justin.tv/video/protocols/hls"
)

// DateTimeFormat defined in section 3.4.5 of HLS Spec
const DateTimeFormat = "2006-01-02T15:04:05"

// DiscontinuityURL for discontinuity tags
const DiscontinuityURL = "discontinuity"

// PlaylistTypeLive, event and Vod to tag a playlist with a type
// https://tools.ietf.org/html/draft-pantos-http-live-streaming-07#section-3.3.7
const (
	PlaylistTypeLive = iota
	PlaylistTypeEvent
	PlaylistTypeVod
)

// StreamOffset is used to mark elapsed time
type StreamOffset struct {
	CurSegIdx   int
	TimeElapsed float64
}

// Chunk holder for video segments
type Chunk struct {
	SequenceNumber  int64
	URL             string
	Duration        float64
	ProgramDateTime *time.Time
}

// Playlist container for a playlist
type Playlist struct {
	Chunks         []Chunk
	StreamDuration float64
	StreamOffset   float64
	TargetDuration int
	IsFinal        bool
	WindowSize     int
	MediaSequence  int
	PlaylistType   int
	InitSegmentUri string
}

func floatSecs(f float64) time.Duration {
	seconds := math.Floor(f)
	return (time.Second * time.Duration(seconds)) + (time.Nanosecond * time.Duration(1e9*(f-seconds)))
}

// ToHLSProto produces a video/protocols/hls protobuf object from a playlist
func (pl Playlist) ToHLSProto(creation time.Time) (out *hls.Playlist, err error) {
	out = &hls.Playlist{
		TargetDuration: int32(pl.TargetDuration),
		MediaSequence:  int32(pl.MediaSequence),
		TimeElapsed:    ptypes.DurationProto(floatSecs(pl.StreamOffset)),
		TimeTotal:      ptypes.DurationProto(floatSecs(pl.StreamDuration)),
		Final:          pl.IsFinal,
	}

	out.Creation, err = ptypes.TimestampProto(creation)
	if err != nil {
		return nil, err
	}

	var lastWasDiscontinuity bool
	for _, chunk := range pl.Chunks {
		if chunk.URL == DiscontinuityURL {
			lastWasDiscontinuity = true
			continue
		}
		segment := &hls.Segment{
			Uri:            chunk.URL,
			SequenceNumber: chunk.SequenceNumber,
			Duration:       ptypes.DurationProto(floatSecs(chunk.Duration / 1000.0)),
			Discontinuity:  lastWasDiscontinuity,
		}
		if chunk.ProgramDateTime != nil {
			var err error
			segment.ProgramDateTime, err = ptypes.TimestampProto(*chunk.ProgramDateTime)
			if err != nil {
				return nil, err
			}
		}
		out.Segments = append(out.Segments, segment)
		lastWasDiscontinuity = false
	}

	return out, nil
}
