package m3u8

import (
	"reflect"
	"testing"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/duration"

	"code.justin.tv/video/protocols/hls"
)

func TestConvertToHLSProto(t *testing.T) {
	now := time.Now()
	nowpb, _ := ptypes.TimestampProto(now)

	type testcase struct {
		in   *Playlist
		want *hls.Playlist
	}
	tests := []testcase{
		testcase{
			in: &Playlist{
				Chunks: []Chunk{
					{1, "1.ts", 2000, nil},
					{2, "2.ts", 2000, nil},
					{3, "3.ts", 2000, &now},
				},
				StreamOffset:   0.0,
				StreamDuration: 6.0,
				TargetDuration: 2,
				IsFinal:        false,
				WindowSize:     10,
				MediaSequence:  1,
				PlaylistType:   PlaylistTypeLive,
			},
			want: &hls.Playlist{
				Segments: []*hls.Segment{
					{Uri: "1.ts", SequenceNumber: 1, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false},
					{Uri: "2.ts", SequenceNumber: 2, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false},
					{Uri: "3.ts", SequenceNumber: 3, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false, ProgramDateTime: nowpb},
				},
				Creation:       nowpb,
				TargetDuration: 2,
				TimeElapsed:    &duration.Duration{Seconds: 0, Nanos: 0},
				TimeTotal:      &duration.Duration{Seconds: 6, Nanos: 0},
				Final:          false,
				MediaSequence:  1,
			},
		},
		testcase{
			in: &Playlist{
				Chunks: []Chunk{
					{4, "4.ts", 2100, nil},
					{5, "5.ts", 2200, nil},
					{6, "6.ts", 100, nil},
				},
				StreamOffset:   6.0,
				StreamDuration: 10.4,
				TargetDuration: 2,
				IsFinal:        false,
				WindowSize:     3,
				MediaSequence:  4,
				PlaylistType:   PlaylistTypeLive,
			},
			want: &hls.Playlist{
				Segments: []*hls.Segment{
					{Uri: "4.ts", SequenceNumber: 4, Duration: &duration.Duration{Seconds: 2, Nanos: 100000000}, Discontinuity: false},
					{Uri: "5.ts", SequenceNumber: 5, Duration: &duration.Duration{Seconds: 2, Nanos: 200000000}, Discontinuity: false},
					{Uri: "6.ts", SequenceNumber: 6, Duration: &duration.Duration{Seconds: 0, Nanos: 100000000}, Discontinuity: false},
				},
				Creation:       nowpb,
				TargetDuration: 2,
				TimeElapsed:    &duration.Duration{Seconds: 6, Nanos: 0},
				TimeTotal:      &duration.Duration{Seconds: 10, Nanos: 400000000},
				Final:          false,
				MediaSequence:  4,
			},
		},
		testcase{
			in: &Playlist{
				Chunks: []Chunk{
					{1, "1.ts", 2000, nil},
					{2, "2.ts", 2000, nil},
					{3, DiscontinuityURL, 0, nil},
					{4, "4.ts", 2000, nil},
					{5, "5.ts", 2000, nil},
				},
				StreamOffset:   0,
				StreamDuration: 8,
				TargetDuration: 2,
				IsFinal:        true,
				WindowSize:     5,
				MediaSequence:  1,
				PlaylistType:   PlaylistTypeLive,
			},
			want: &hls.Playlist{
				Segments: []*hls.Segment{
					{Uri: "1.ts", SequenceNumber: 1, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false},
					{Uri: "2.ts", SequenceNumber: 2, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false},
					{Uri: "4.ts", SequenceNumber: 4, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: true},
					{Uri: "5.ts", SequenceNumber: 5, Duration: &duration.Duration{Seconds: 2, Nanos: 0}, Discontinuity: false},
				},
				Creation:       nowpb,
				TargetDuration: 2,
				TimeElapsed:    &duration.Duration{Seconds: 0, Nanos: 0},
				TimeTotal:      &duration.Duration{Seconds: 8, Nanos: 0},
				Final:          true,
				MediaSequence:  1,
			},
		},
	}
	for i, tc := range tests {
		have, _ := tc.in.ToHLSProto(now)
		if !reflect.DeepEqual(have, tc.want) {
			t.Errorf("case %d: %+v.ToHLSProto(%s) wrong", i+1, tc.in, now)
			t.Logf("have=%+v", have)
			t.Logf("want=%+v", tc.want)
		}
	}
}
