package noisy

import (
	"fmt"
	"log"
	"strings"

	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"
	"code.justin.tv/video/gotranscoder/pkg/statsd"

	"github.com/hashicorp/consul/api"
	"github.com/stvp/rollbar"
)

// HostInfo for logging purposes
type HostInfo struct {
	Host        string
	Pop         string
	Environment string
}

// Config sets the logging configuration
type Config struct {
	HostInfo         *HostInfo
	ESClient         *elasticsearch.Settings
	RollbarToken     string
	TranscodeProfile string
	Channel          string
	ChannelID        uint64
}

var cfg *Config

// Configure sets the global logging configuration based on passed in config
func Configure(_cfg *Config, consulAddress string) {
	if _cfg.HostInfo == nil {
		_cfg.HostInfo = getHostInfo(consulAddress, "video-workers")
	}
	if _cfg.TranscodeProfile == "" {
		_cfg.TranscodeProfile = "unknown"
	}
	rollbar.Token = _cfg.RollbarToken
	rollbar.Environment = _cfg.HostInfo.Environment
	cfg = _cfg
}

// Error emits an error to each of the logging mechanisms
func Error(stat, message string) {
	if cfg == nil {
		log.Printf("noisy.Error() called but not configured")
		return
	}
	statsd.Inc(fmt.Sprintf("error.%s.%s.%s.%s.%s", cfg.HostInfo.Pop, cfg.HostInfo.Environment,
		cfg.HostInfo.Host, cfg.TranscodeProfile, stat), 1, 1.0)
	rollbar.Error(rollbar.WARN, fmt.Errorf(message),
		&rollbar.Field{Name: "Message", Data: message},
		&rollbar.Field{Name: "Stat", Data: stat},
		&rollbar.Field{Name: "Host", Data: cfg.HostInfo.Host},
		&rollbar.Field{Name: "Pop", Data: cfg.HostInfo.Pop},
		&rollbar.Field{Name: "Environment", Data: cfg.HostInfo.Environment},
		&rollbar.Field{Name: "TranscodeProfile", Data: cfg.TranscodeProfile},
		&rollbar.Field{Name: "ChannelID", Data: cfg.ChannelID},
		&rollbar.Field{Name: "Channel:", Data: cfg.Channel},
	)

	ESWrite(stat, "Error", message)
}

// ESWrite writes logs to ElasticSearch
func ESWrite(stat, messageType, message string) {
	if cfg == nil {
		log.Printf("noisy.ESWrite() called but not configured")
		return
	}
	if cfg.ESClient != nil {
		cfg.ESClient.WriteLog(&elasticsearch.Log{
			MessageType: messageType,
			Channel:     cfg.Channel,
			ChannelID:   cfg.ChannelID,
			Data: map[string]interface{}{
				"Message":          message,
				"Stat":             stat,
				"Host":             cfg.HostInfo.Host,
				"Pop":              cfg.HostInfo.Pop,
				"Environment":      cfg.HostInfo.Environment,
				"TranscodeProfile": cfg.TranscodeProfile,
			}}, false)
	}
}

func getHostInfo(consulAddress, serviceName string) *HostInfo {
	hostInfo := &HostInfo{
		Host:        "unknown",
		Pop:         "unknown",
		Environment: "unknown",
	}
	consulConfig := api.DefaultConfig()
	consulConfig.Address = consulAddress
	client, err := api.NewClient(consulConfig)
	if err != nil {
		log.Printf("Failed to create consul client - cannot use debug stats: %s", err)
		return hostInfo
	}
	self, err := client.Agent().Self()
	if err != nil {
		log.Printf("Failed to get consul self: %s", err)
		return hostInfo
	}
	var nodeID string
	if config, ok := self["Config"]; ok {
		if pop, ok := config["Datacenter"].(string); ok {
			hostInfo.Pop = pop
		}
		if _nodeID, ok := config["NodeID"].(string); ok {
			nodeID = _nodeID
		}
	}
	if member, ok := self["Member"]; ok {
		if host, ok := member["Name"].(string); ok {
			hostInfo.Host = host
		}
	}
	catalogNode, _, err := client.Catalog().Node(nodeID, &api.QueryOptions{AllowStale: true})
	if err != nil || catalogNode == nil {
		return hostInfo
	}
	if service, ok := catalogNode.Services[serviceName]; ok {
		hostInfo.Environment = getBestTag(service.Tags)
	}
	return hostInfo
}

func getBestTag(tags []string) string {
	if len(tags) == 0 {
		return "unknown"
	}
	if len(tags) == 1 {
		return tags[0]
	}
	for _, tag := range tags {
		if !strings.Contains(strings.ToLower(tag), "prod") {
			return tag
		}
	}
	return tags[0]
}
