package noisy

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"net/url"
	"reflect"
	"strconv"
	"strings"
	"testing"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"

	"github.com/stvp/rollbar"
)

func TestConfigure(t *testing.T) {
	var rollbarChan = make(chan bool, 1)
	var esChan = make(chan bool, 1)
	consulMux := http.NewServeMux()
	consulMux.HandleFunc("/v1/agent/self", func(w http.ResponseWriter, r *http.Request) {
		http.ServeFile(w, r, "testdata/self.json")
	})
	consulMux.HandleFunc("/v1/catalog/node/5e8eba8f-55c0-41d1-ad74-d4c27e1c460b", func(w http.ResponseWriter, r *http.Request) {
		http.ServeFile(w, r, "testdata/catalognode.json")
	})
	consulServer := httptest.NewServer(consulMux)
	consulURL, _ := url.Parse(consulServer.URL)
	consulPort, _ := strconv.Atoi(strings.Split(consulURL.Host, ":")[1])

	rollbarMux := http.NewServeMux()
	rollbarMux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		if r.Method != "POST" {
			return
		}
		result := make(map[string]interface{})
		err := json.NewDecoder(r.Body).Decode(&result)
		data := result["data"].(map[string]interface{})
		if data["Stat"] != "test.error" || data["title"] != "a test error" {
			fmt.Println("received bad rollbar post:", result, err)
			return
		}
		select {
		case rollbarChan <- true:
			fmt.Println("Calling rallbar handler the first time")
		default:
			fmt.Println("Calling rollbar handler more than once")
		}
	})
	rollbarServer := httptest.NewServer(rollbarMux)

	esMux := http.NewServeMux()
	esMux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		if r.Method != "POST" {
			return
		}
		result := make(map[string]interface{})
		err := json.NewDecoder(r.Body).Decode(&result)
		data := result["Error"].(map[string]interface{})
		if data["Stat"] != "test.error" || data["Message"] != "a test error" {
			fmt.Println("received bad elasticsearch post:", result, err)
			return
		}
		select {
		case esChan <- true:
			fmt.Println("Calling ES handler the first time")
		default:
			fmt.Println("Calling ES handler more than once")
		}

	})
	esServer := httptest.NewServer(esMux)
	esURL, _ := url.Parse(esServer.URL)
	esPort, _ := strconv.Atoi(strings.Split(esURL.Host, ":")[1])

	expectedCfg := &Config{
		HostInfo: &HostInfo{
			Host:        "transcode-x2-test-8c74d4",
			Pop:         "sjc02",
			Environment: "transcoder-testing-a",
		},
		RollbarToken:     "rollbar-token",
		TranscodeProfile: "high-quality-profile",
		Channel:          "cool-channel",
		ChannelID:        12345,
	}

	inputCfg := &Config{
		RollbarToken:     "rollbar-token",
		TranscodeProfile: "high-quality-profile",
		Channel:          "cool-channel",
		ChannelID:        12345,
	}

	Configure(inputCfg, fmt.Sprintf("localhost:%d", consulPort))

	if !reflect.DeepEqual(cfg, expectedCfg) {
		t.Errorf("cfg != expectedCfg:\n%+v\n%+v", cfg, expectedCfg)
	}

	rollbar.Endpoint = rollbarServer.URL
	cfg.ESClient = &elasticsearch.Settings{
		Host:       "localhost",
		Port:       int32(esPort),
		SampleRate: 1.0,
	}
	cfg.ESClient.Initialize()

	Error("test.error", "a test error")
	timeoutChan := time.After(10 * time.Millisecond)

	rollbarOK := false
	esOK := false

	select {
	case rollbarOK = <-rollbarChan:
	case <-timeoutChan:
	}

	select {
	case esOK = <-esChan:
	case <-timeoutChan:
	}

	if !rollbarOK {
		t.Error("bad rollbar post")
	}

	if !esOK {
		t.Error("bad elasticsearch post")
	}
}
