package notify

import (
	"encoding/json"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"
	streamlog "code.justin.tv/video/streamlog/pkg/wswriter"
)

const (
	lvsStreamReadySubject   = "StreamReady"
	lvsStreamDownSubject    = "StreamDown"
	lvsVodStartSubject      = "VodStart"
	lvsVodStopSubject       = "VodStop"
	lvsLiveStreamEndSubject = "LiveStreamEnd"
)

// LvsNotifySettings holds the SNS configuration for broadcast up/down notifications
type LvsNotifySettings struct {
	sns        SNS
	topicArn   string
	channel    string
	es         *elasticsearch.Settings
	ContentId  string `json:"content_id"`
	CustomerId string `json:"customer_id"`
	Timestamp  int64  `json:"timestamp"`
	SessionId  string `json:"session_id"`
	S3Filename string `json:"s3_filename,omitempty"`
}

// InitLvsNotify sets up a LvsNotifySettings struct according to the passed in configuration
func InitLvsNotify(customerId, contentId, sessionId, topicArn, channel string, es *elasticsearch.Settings) Notifier {
	return &LvsNotifySettings{
		sns:        NewSNS(nil, ""),
		topicArn:   topicArn,
		channel:    channel,
		ContentId:  contentId,
		CustomerId: customerId,
		SessionId:  sessionId,
		es:         es,
	}
}

// NotifyStreamReady emits a stream ready notification to SNS
func (n *LvsNotifySettings) NotifyStreamReady() error {
	n.Timestamp = time.Now().Unix()
	msg, _ := json.Marshal(n)
	err := n.sns.Publish(string(msg), lvsStreamReadySubject, n.topicArn)
	if err == nil {
		streamlog.Log(n.channel, "sns_notification", lvsStreamReadySubject)
	}
	logSnsNotificationES(n.es, n.topicArn, n.channel, n.SessionId, lvsStreamReadySubject, err)
	return err
}

// NotifyStreamDown emits a stream down notification to SNS
func (n *LvsNotifySettings) NotifyStreamDown() error {
	n.Timestamp = time.Now().Unix()
	msg, _ := json.Marshal(n)
	err := n.sns.Publish(string(msg), lvsStreamDownSubject, n.topicArn)
	if err == nil {
		streamlog.Log(n.channel, "sns_notification", lvsStreamDownSubject)
	}
	logSnsNotificationES(n.es, n.topicArn, n.channel, n.SessionId, lvsStreamDownSubject, err)

	return err
}

// NotifyVodStart emits a VOD start notification to SNS
func (n *LvsNotifySettings) NotifyVodStart(s3Filename string) error {
	s := LvsNotifySettings{
		ContentId:  n.ContentId,
		CustomerId: n.CustomerId,
		SessionId:  n.SessionId,
	}
	s.Timestamp = time.Now().Unix()
	s.S3Filename = s3Filename
	msg, _ := json.Marshal(s)
	err := n.sns.Publish(string(msg), lvsVodStartSubject, n.topicArn)
	if err == nil {
		streamlog.Log(n.channel, "sns_notification", lvsVodStartSubject)
	}
	logSnsNotificationES(n.es, n.topicArn, n.channel, n.SessionId, lvsVodStartSubject, err)

	return err
}

// NotifyVodStop emits a VOD stop notification to SNS
func (n *LvsNotifySettings) NotifyVodStop(s3Filename string) error {
	s := LvsNotifySettings{
		ContentId:  n.ContentId,
		CustomerId: n.CustomerId,
		SessionId:  n.SessionId,
	}
	s.Timestamp = time.Now().Unix()
	s.S3Filename = s3Filename
	msg, _ := json.Marshal(s)
	err := n.sns.Publish(string(msg), lvsVodStopSubject, n.topicArn)
	if err == nil {
		streamlog.Log(n.channel, "sns_notification", lvsVodStopSubject)
	}
	logSnsNotificationES(n.es, n.topicArn, n.channel, n.SessionId, lvsVodStopSubject, err)
	return err
}

// NotifyLiveStreamEnd emits a stream end notification to SNS
func (n *LvsNotifySettings) NotifyLiveStreamEnd() error {
	n.Timestamp = time.Now().Unix()
	msg, _ := json.Marshal(n)
	err := n.sns.Publish(string(msg), lvsLiveStreamEndSubject, n.topicArn)
	if err == nil {
		streamlog.Log(n.channel, "sns_notification", lvsLiveStreamEndSubject)
	}
	logSnsNotificationES(n.es, n.topicArn, n.channel, n.SessionId, lvsLiveStreamEndSubject, err)
	return err
}

func logSnsNotificationES(es *elasticsearch.Settings, topic, channel, sessionId, subject string, err error) {
	esData := map[string]interface{}{
		"SnsEndpoint": topic,
		"Subject":     subject,
		"Error":       err,
		"SessionID":   sessionId,
	}
	var msgType string
	if err != nil {
		msgType = "SnsNotificationFailure"
	} else {
		msgType = "SnsNotificationSuccess"
	}

	es.WriteLog(&elasticsearch.Log{
		MessageType: msgType,
		Channel:     channel,
		Data:        esData,
	}, false)

}
