package notify

import (
	"encoding/json"
	"fmt"
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/video/gotranscoder/pkg/notify/fakes"
)

func TestLvsNotifyStreamUp(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
		SessionId:  "test-session",
	}

	err := settings.NotifyStreamReady()
	assert.Nil(err)

	// verify SNS call counts and args
	assert.Equal(1, svc.PublishCallCount())

	publishInput := svc.PublishArgsForCall(0)
	assert.Equal("StreamReady", *publishInput.Subject)

	result := &LvsNotifySettings{}
	msg := *publishInput.Message
	err = json.Unmarshal([]byte(msg), result)
	assert.Nil(err)

	assert.Equal("Twitch", result.CustomerId)
	assert.Equal("test-content", result.ContentId)
	assert.Equal("test-session", result.SessionId)
}

func TestLvsNotifyStreamDown(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
		SessionId:  "test-session",
	}

	err := settings.NotifyStreamDown()
	assert.Nil(err)

	// verify SNS call counts and args
	assert.Equal(1, svc.PublishCallCount())

	publishInput := svc.PublishArgsForCall(0)
	assert.Equal("StreamDown", *publishInput.Subject)

	result := &LvsNotifySettings{}
	msg := *publishInput.Message
	err = json.Unmarshal([]byte(msg), result)
	assert.Nil(err)

	assert.Equal("Twitch", result.CustomerId)
	assert.Equal("test-content", result.ContentId)
	assert.Equal("test-session", result.SessionId)
}

func TestLvsNotifyStreamUpFail(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}
	svc.PublishReturns(nil, fmt.Errorf("test-error"))

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
	}

	err := settings.NotifyStreamReady()
	assert.NotNil(err)
}

func TestLvsNotifyStreamDownFail(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	svc.PublishReturns(nil, fmt.Errorf("test-error"))

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
	}

	err := settings.NotifyStreamDown()
	assert.NotNil(err)
}

func TestLvsNotifyVodStart(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
		SessionId:  "test-session",
	}

	err := settings.NotifyVodStart("test-s3-filename")
	assert.Nil(err)

	// verify SNS call counts and args
	assert.Equal(1, svc.PublishCallCount())

	publishInput := svc.PublishArgsForCall(0)
	assert.Equal("VodStart", *publishInput.Subject)

	result := &LvsNotifySettings{}
	msg := *publishInput.Message
	err = json.Unmarshal([]byte(msg), result)
	assert.Nil(err)

	assert.Equal("Twitch", result.CustomerId)
	assert.Equal("test-content", result.ContentId)
	assert.Equal("test-session", result.SessionId)
	assert.Equal("test-s3-filename", result.S3Filename)
}

func TestLvsNotifyVodStop(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
		SessionId:  "test-session",
	}

	err := settings.NotifyVodStop("test-s3-filename")
	assert.Nil(err)

	// verify SNS call counts and args
	assert.Equal(1, svc.PublishCallCount())

	publishInput := svc.PublishArgsForCall(0)
	assert.Equal("VodStop", *publishInput.Subject)

	result := &LvsNotifySettings{}
	msg := *publishInput.Message
	err = json.Unmarshal([]byte(msg), result)
	assert.Nil(err)

	assert.Equal("Twitch", result.CustomerId)
	assert.Equal("test-content", result.ContentId)
	assert.Equal("test-session", result.SessionId)
	assert.Equal("test-s3-filename", result.S3Filename)
}

func TestLvsNotifyLiveStreamEnd(t *testing.T) {
	assert := assert.New(t)
	svc := &fakes.FakeSNSAPI{}

	settings := &LvsNotifySettings{
		sns:        NewSNS(svc, "lvs.test"),
		topicArn:   "test-sns",
		CustomerId: "Twitch",
		ContentId:  "test-content",
		SessionId:  "test-session",
	}

	err := settings.NotifyLiveStreamEnd()
	assert.Nil(err)

	// verify SNS call counts and args
	assert.Equal(1, svc.PublishCallCount())

	publishInput := svc.PublishArgsForCall(0)
	assert.Equal("LiveStreamEnd", *publishInput.Subject)

	result := &LvsNotifySettings{}
	msg := *publishInput.Message
	err = json.Unmarshal([]byte(msg), result)
	assert.Nil(err)

	assert.Equal("Twitch", result.CustomerId)
	assert.Equal("test-content", result.ContentId)
	assert.Equal("test-session", result.SessionId)

}
