package notify

import (
	"encoding/json"
	"fmt"

	"code.justin.tv/video/gotranscoder/pkg/elasticsearch"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
)

const (
	// old integration
	streamReadyTopicARN = "arn:aws:sns:us-west-2:848744099708:TwitchStreamReadyNotifications"
	streamDownTopicARN  = "arn:aws:sns:us-west-2:848744099708:TwitchStreamDownNotifications"
	streamReadySubject  = "StreamReady"
	streamDownSubject   = "StreamDown"

	// new integration
	notificationsStreamEventSubject = "StreamEvent" // not really a part of the spec
	notificationsStreamReadyEvent   = "ready"
	notificationsStreamDownEvent    = "down"
)

// Notifier is an interface that holds StreamReady and StreamDown notifications
type Notifier interface {
	NotifyStreamReady() error
	NotifyStreamDown() error
	NotifyVodStart(string) error
	NotifyVodStop(string) error
	NotifyLiveStreamEnd() error
}

// NotifySettings holds the SNS configuration for broadcast up/down notifications
type NotifySettings struct {
	oldSNS       SNS
	newSNS       SNS
	ChannelID    int    `json:"channel_id"`
	Channel      string `json:"channel"`
	StreamID     int    `json:"stream_id"`
	SessionID    string `json:"session_id"`
	BroadcastTag string `json:"broadcast_tag,omitempty"`

	// The topic new-style notifications are sent to. Staging is forwarded to Production by the backend
	notificationsStreamEventTopicArn string
}

// NotificationsMessage adds additional "event" property since events are merged into one stream in the new notifications system
type NotificationsMessage struct {
	NotifySettings
	Event string `json:"event"` // "ready" or "down"
}

// Init sets up a NotifySettings struct according to the passed in configuration
func Init(channel string, channelID int, streamID int, sessionID, broadcastTag,
	notificationsStreamEventTopicArn string, snsImpl snsiface.SNSAPI) Notifier {
	return &NotifySettings{
		oldSNS:       NewSNS(snsImpl, ".video"),
		newSNS:       NewSNS(snsImpl, ".notifications"),
		ChannelID:    channelID,
		Channel:      channel,
		StreamID:     streamID,
		SessionID:    sessionID,
		BroadcastTag: broadcastTag,

		notificationsStreamEventTopicArn: notificationsStreamEventTopicArn,
	}
}

// NotifyStreamReady emits a stream ready notification to SNS
func (n *NotifySettings) NotifyStreamReady() error {
	oldErr := encodeAndPublish(n.oldSNS, n, streamReadySubject, streamReadyTopicARN)
	newErr := encodeAndPublish(n.newSNS, &NotificationsMessage{
		NotifySettings: *n,
		Event:          notificationsStreamReadyEvent,
	}, notificationsStreamEventSubject, n.notificationsStreamEventTopicArn)
	if oldErr != nil || newErr != nil {
		return fmt.Errorf("old SNS error: %v, new SNS error: %v", oldErr, newErr)
	}
	return nil
}

// NotifyStreamDown emits a stream down notification to SNS
func (n *NotifySettings) NotifyStreamDown() error {
	oldErr := encodeAndPublish(n.oldSNS, n, streamDownSubject, streamDownTopicARN)
	newErr := encodeAndPublish(n.newSNS, &NotificationsMessage{
		NotifySettings: *n,
		Event:          notificationsStreamDownEvent,
	}, notificationsStreamEventSubject, n.notificationsStreamEventTopicArn)
	if oldErr != nil || newErr != nil {
		return fmt.Errorf("old SNS error: %v, new SNS error: %v", oldErr, newErr)
	}
	return nil
}

// NotifyVodStart emits a VOD start notification to SNS. Not implemented
func (n *NotifySettings) NotifyVodStart(s3Filename string) error {
	return nil
}

// NotifyVodStop emits a VOD stop notification to SNS. Not Implemented
func (n *NotifySettings) NotifyVodStop(s3Filename string) error {
	return nil
}

// NotifyLiveStreamEnd emits a notification when gotranscoder exits. Not Implemented
func (n *NotifySettings) NotifyLiveStreamEnd() error {
	return nil
}

// InitNotifier initializes notifier based on parameters
func InitNotifier(channel string, channelID, destination int, sessionID, broadcastTag, customerID, contentID, topicArn string, isLvsChannel bool, es *elasticsearch.Settings) Notifier {
	//Init Notify
	var notifyObj Notifier
	if isLvsChannel {
		notifyObj = InitLvsNotify(customerID, contentID, sessionID, topicArn, channel, es)
	} else {
		notifyObj = Init(channel, channelID, destination, sessionID, broadcastTag, topicArn, nil)
	}

	return notifyObj

}

// helper for all the encoding that is going on during transition to new notifications backend
func encodeAndPublish(sns SNS, message interface{}, subject, topicARN string) error {
	msg, err := json.Marshal(message)
	if err != nil {
		return err
	}
	return sns.Publish(string(msg), subject, topicARN)
}
