package origin

import (
	"fmt"
	"log"
	"net/http"

	"context"

	"code.justin.tv/video/origin/rpc/originctl"
)

// HLSURLBase returns the URL prefix for all data requests to the origin server
// for a given session. This value is passed in to TwitchTranscoder at runtime.
func HLSURLBase(endpoint string, sess *originctl.Session) string {
	if sess == nil {
		return ""
	}
	return fmt.Sprintf("%s/streams/%s", endpoint, sess.Id)
}

// Origin service interface
type Origin interface {
	StartSession(ctx context.Context, channelID string, variants []string) (*originctl.Session, func(), error)
}

type origin struct {
	cfg      Config
	client   originctl.OriginControlV1
	sessions map[string]*originctl.Session
}

// Config holds configuration for access to the origin service
type Config struct {
	Endpoint string
}

// New constructs a new origin service wrapper based on the input configuration
func New(cfg Config) Origin {
	return &origin{
		cfg:      cfg,
		client:   originctl.NewOriginControlV1JSONClient(cfg.Endpoint, &http.Client{}),
		sessions: make(map[string]*originctl.Session),
	}
}

// StartSession initiates a new origin session
func (o *origin) StartSession(ctx context.Context, channelID string, variants []string) (*originctl.Session, func(), error) {
	resp, err := o.client.StartSession(ctx, &originctl.StartSessionRequest{
		Stream: &originctl.Stream{
			ChannelId: channelID,
			Variants:  variants,
		},
	})
	if err != nil {
		log.Println("[ORIGIN] Session.Start() failed:", err)
		return nil, nil, err
	}
	log.Println("[ORIGIN] Session started:", resp.Session.Id)
	o.sessions[channelID] = resp.Session
	exitCallback := func() {
		if err := o.stop(context.Background(), channelID); err != nil {
			log.Println("[ORIGIN] failed to gracefully stop origin session:", err)
		}
	}
	return resp.Session, exitCallback, nil
}

func (o *origin) stop(ctx context.Context, channelID string) error {
	session, ok := o.sessions[channelID]
	if !ok || session == nil {
		return fmt.Errorf("can't stop: no active session for channel id %s", channelID)
	}
	_, err := o.client.TerminateSession(ctx, &originctl.TerminateSessionRequest{
		Id: session.Id,
	})
	if err != nil {
		return fmt.Errorf("TerminateSession failed: %v", err)
	}
	log.Println("[ORIGIN] Session terminated successfully")
	return nil
}
