package origin_test

import (
	"context"
	"errors"
	"net/http/httptest"
	"strings"

	. "code.justin.tv/video/gotranscoder/pkg/origin"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/origin/rpc/originctl/originctlfakes"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"testing"
)

func TestOriginWorking(t *testing.T) {
	var (
		control = &originctlfakes.FakeOriginControlV1{}
		server  = httptest.NewServer(originctl.NewOriginControlV1Server(control, nil))
		client  = New(Config{Endpoint: server.URL})

		goodChannelId = "asdf-zxcv"
		goodSessionId = "12345"
		goodSession   = &originctl.Session{Id: goodSessionId}
	)

	defer server.Close()

	control.StartSessionReturns(&originctl.StartSessionResponse{
		Session: goodSession,
	}, nil)
	control.TerminateSessionReturns(&originctl.TerminateSessionResponse{}, nil)

	session, exitFunction, err := client.StartSession(context.Background(), goodChannelId, []string{"chunked"})
	require.NoError(t, err)
	require.NotNil(t, exitFunction)
	assert.Equal(t, goodSession, session)
	assert.Equal(t, 0, control.TerminateSessionCallCount())

	exitFunction()
	assert.Equal(t, 1, control.TerminateSessionCallCount())
	_, terminatedSession := control.TerminateSessionArgsForCall(0)
	assert.Equal(t, goodSessionId, terminatedSession.Id)
}

func TestOriginDown(t *testing.T) {
	var (
		control = &originctlfakes.FakeOriginControlV1{}
		server  = httptest.NewServer(originctl.NewOriginControlV1Server(control, nil))
		client  = New(Config{Endpoint: server.URL})

		goodChannelId = "asdf-zxcv"
	)

	defer server.Close()

	alwaysErr := errors.New("always errors")
	control.StartSessionReturns(nil, alwaysErr)

	session, exitFunction, err := client.StartSession(context.Background(), goodChannelId, []string{"chunked"})
	assert.Error(t, err)
	assert.True(t, strings.Contains(err.Error(), alwaysErr.Error()))
	assert.Nil(t, exitFunction)
	assert.Nil(t, session)
	assert.Equal(t, 0, control.TerminateSessionCallCount())
}
