package pluginloop

import (
	"errors"
	"log"
	"sync"
	"time"
)

// ConcurrentPlugin is the container for a list of plugins that get processed concurrently.
type ConcurrentPlugin struct {
	sync.Mutex
	EvalProcessingTimeout time.Duration
	registeredPlugins     map[string]Plugin
}

// Plugin interface to implement a plugin.
type Plugin interface {
	Process(interface{}, <-chan struct{}) error
	Name() string
	Initialize()
}

// Register adds a plugin into the processing queue.
// Only allow one plugin of a type by name
func (cp *ConcurrentPlugin) Register(plugin Plugin) error {
	cp.Lock()
	defer cp.Unlock()
	if cp.registeredPlugins == nil {
		cp.registeredPlugins = make(map[string]Plugin)

	}

	if _, ok := cp.registeredPlugins[plugin.Name()]; ok {
		return errors.New("Plugin is already registered")
	}

	cp.registeredPlugins[plugin.Name()] = plugin
	log.Printf("Plugin registered: %s", plugin.Name())

	return nil
}

// Unregister removes a plugin from the processing queue
func (cp *ConcurrentPlugin) Unregister(plugin Plugin) {
	cp.Lock()
	defer cp.Unlock()
	delete(cp.registeredPlugins, plugin.Name())
}

// Take out the plugin executing to allow different timeouts on different segments
func (cp *ConcurrentPlugin) executor(p Plugin, o *interface{}, pc <-chan struct{}, c chan error) {
	err := p.Process(*o, pc)
	if err != nil {
		log.Printf("EXECUTOR: Error executing Process of %s : %s ", p.Name(), err)
	}
	c <- err
}

// Eval evaluates the process methods in each plugin registered in the
// processing queue.
// The idea is to have several plusing respond to some generated datum as an event:
// Incoming object X -> Plugin Pipeline -> All plugins evaluate X
func (cp *ConcurrentPlugin) Eval(obj *interface{}) {
	var wg sync.WaitGroup

	cp.Lock()
	var currentPlugins = make(map[string]Plugin)
	for k, v := range cp.registeredPlugins {
		currentPlugins[k] = v
	}
	cp.Unlock()

	// Timeout Callback Channel
	timeout := make(chan struct{})

	t := time.AfterFunc(cp.EvalProcessingTimeout, func() { close(timeout) })
	defer t.Stop()

	for _, plugin := range currentPlugins {
		//log.Printf("[PLUGIN PROCESSING]: %v\n on %+v", key, obj)
		wg.Add(1)

		go func(p Plugin) {
			defer wg.Done()

			c := make(chan error, 1)

			go cp.executor(p, obj, timeout, c)

			select {
			case err := <-c:
				if err != nil {
					log.Printf("Error in Plugin [%s] Process: %s", p.Name(), err)
				}
			case <-timeout:
				log.Printf("[PLUGING PROCESSING] Plugin [%s] Process Timeout", p.Name())
			}

		}(plugin)
	}

	wg.Wait()
}
