package pluginloop

import (
	"errors"
	"log"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

type TestPlugin struct {
}

type TestPlug struct {
	Processed            bool
	ProcessShouldErr     bool
	ProcessShouldTimeout bool
	InputReceived        string
	ID                   string
}

func (tp *TestPlug) Name() string {
	return tp.ID
}

func (tp *TestPlug) Initialize() {}

// Simulate a process.
// Either succeed, error, or timeout depending on flags
func (tp *TestPlug) Process(input interface{}, timeoutCallback <-chan struct{}) error {

	if tp.ProcessShouldTimeout {
		log.Println("Sleeping Test process....")
		time.Sleep(500 * time.Second) // should be handled by plugin loop
	}

	if tp.ProcessShouldErr {
		return errors.New("test error")
	}

	tp.Processed = true
	tp.InputReceived = input.(string)

	return nil
}

func getData() interface{} {
	return "test"
}

func TestRegistration(t *testing.T) {
	var (
		testPluginContainer = ConcurrentPlugin{
			EvalProcessingTimeout: 100,
		}
		// Register one plugin
		t2 = TestPlug{
			Processed:        false,
			ProcessShouldErr: false,
			ID:               "testRegistration",
		}

		assert = assert.New(t)
	)

	err := testPluginContainer.Register(&t2)
	assert.NoError(err)
	assert.NotNil(testPluginContainer.registeredPlugins["testRegistration"])

	// registering the same should err
	assert.Error(testPluginContainer.Register(&t2))

	// deregister the plugin
	// registering the same should err
	testPluginContainer.Unregister(&t2)
	assert.Nil(testPluginContainer.registeredPlugins["testRegistration"])

	// deregister twice
	testPluginContainer.Unregister(&t2)
}

func TestProcess(t *testing.T) {
	var (
		assert              = assert.New(t)
		data                = getData()
		testPluginContainer = ConcurrentPlugin{
			EvalProcessingTimeout: 1000 * time.Millisecond,
		}

		// Register two plugins
		t1 = TestPlug{
			Processed:        false,
			ProcessShouldErr: false,
			ID:               "concurrentOne",
		}
		t2 = TestPlug{
			Processed:        false,
			ProcessShouldErr: false,
			ID:               "concurrentTwo",
		}
	)

	assert.NoError(testPluginContainer.Register(&t1))
	assert.NoError(testPluginContainer.Register(&t2))

	testPluginContainer.Eval(&data)

	assert.Equal(t1.InputReceived, data)
	assert.Equal(t2.InputReceived, data)
}

func TestErrorHandler(t *testing.T) {
	var (
		assert              = assert.New(t)
		data                = getData()
		testPluginContainer = ConcurrentPlugin{
			EvalProcessingTimeout: 1000 * time.Millisecond,
		}
		// plugin should error
		t1 = TestPlug{
			Processed:        false,
			ProcessShouldErr: true,
			ID:               "erroredProcess",
		}
	)

	// Register plugins that will error out
	testPluginContainer.Register(&t1)
	testPluginContainer.Eval(&data)

	assert.NotEqual(t1.InputReceived, data)
	assert.False(t1.Processed)
}

func TestTimeoutHandler(t *testing.T) {
	var (
		assert = assert.New(t)
		data   = getData()
		// plugin will sleep and timeout
		testPluginContainer = ConcurrentPlugin{
			EvalProcessingTimeout: 1 * time.Millisecond,
		}

		t1 = TestPlug{
			Processed:            false,
			ProcessShouldErr:     false,
			ProcessShouldTimeout: true,
			ID:                   "timeoutProcess1",
		}
		t2 = TestPlug{
			Processed:            false,
			ProcessShouldErr:     false,
			ProcessShouldTimeout: true,
			ID:                   "timeoutProcess2",
		}
		t3 = TestPlug{
			Processed:            false,
			ProcessShouldErr:     false,
			ProcessShouldTimeout: true,
			ID:                   "timeoutProcess3",
		}
	)

	// Plugin that times out
	testPluginContainer.Register(&t1)
	testPluginContainer.Register(&t2)
	testPluginContainer.Register(&t3)
	testPluginContainer.Eval(&data)
	assert.NotEqual(t1.InputReceived, data)
	assert.False(t1.Processed)
}
