package spade

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"golang.org/x/net/context"
)

const (
	// TranscodeStreamStartEventName is the event name which will be used in spade
	TranscodeStreamStartEventName = "transcode_stream_start"
	// SpadeEventSuccess stat for a successful request to spade
	SpadeEventSuccess = "spade.success"
	// SpadeEventFailure stat for a failure in sending event to spade
	SpadeEventFailure = "spade.failure"
)

// WrapperClient wraps around the common spade client, mainly to make testing easier
type WrapperClient struct {
	client spade.Client
}

// TranscodeStreamStart is the struct which defines the transcode_stream_start events properties
type TranscodeStreamStart struct {
	Channel         string `json:"channel,omitempty"`
	ChannelId       uint64 `json:"channel_id,omitempty"`
	CustomerId      string `json:"customer_id"`
	ContentId       string `json:"content_id"`
	TranscodeType   string `json:"transcode_type,omitempty"`
	StartTime       int64  `json:"start_time,omitempty"`
	IngestSessionId string `json:"ingest_session_id,omitempty"`
	ArchivesEnabled bool   `json:"archive_enabled"`
	StreamId        int64  `json:"stream_id,omitempty"`
}

// GetSpadeClient constructs a spade client given the spade url and schema
func GetSpadeClient(baseUrl, scheme string) (WrapperClient, error) {
	client, err := NewSpadeClient(baseUrl, scheme)
	if err != nil {
		return WrapperClient{}, err
	}

	return NewWrapperClient(client), nil
}

// NewSpadeClient constructs the spade client using the common spade client library
func NewSpadeClient(spadeUrl string, scheme string) (spade.Client, error) {
	u := url.URL{
		Scheme: scheme,
		Host:   spadeUrl,
		Path:   "/",
	}

	httpClient := &http.Client{
		Timeout: 5 * time.Second,
	}
	c, err := spade.NewClient(
		spade.InitHTTPClient(httpClient),
		spade.InitBaseURL(u),
		spade.InitMaxConcurrency(10),
	)
	if err != nil {
		return nil, err
	}

	return c, nil
}

// NewWrapperClient is a wrapper around the common spade client library
func NewWrapperClient(coreClient spade.Client) WrapperClient {
	return WrapperClient{
		client: coreClient,
	}
}

func (wc *WrapperClient) sendEvent(eventName string, properties interface{}) error {
	if wc.client == nil {
		return fmt.Errorf("Spade Client is not set")
	}

	return wc.client.TrackEvent(context.Background(), eventName, properties)
}

// SendStreamStartEvent will be called with all the properties to send the transcode_stream_start event
func (wc *WrapperClient) SendStreamStartEvent(channel, customerID, contentID, sessionID, transcodeProfile string,
	channelID uint64, startTime, streamID int64, archiveEnabled bool) {
	transcodestreamStart := TranscodeStreamStart{
		Channel:         channel,
		ChannelId:       channelID,
		CustomerId:      customerID,
		ContentId:       contentID,
		TranscodeType:   transcodeProfile,
		StreamId:        streamID,
		StartTime:       startTime,
		ArchivesEnabled: archiveEnabled,
		IngestSessionId: sessionID,
	}
	err := wc.sendEvent(TranscodeStreamStartEventName, transcodestreamStart)
	if err != nil {
		statsd.Inc(SpadeEventFailure, 1, 1)
	} else {
		statsd.Inc(SpadeEventSuccess, 1, 1)
	}

}
