package spade

import (
	"testing"

	"strings"

	"fmt"

	"code.justin.tv/common/spade-client-go/spade/spadefakes"
	"github.com/stretchr/testify/require"
)

func TestEmptyClient(t *testing.T) {
	wrapper := &WrapperClient{
		client: nil,
	}

	err := wrapper.sendEvent("test", "test")
	require.NotNil(t, err)
	require.True(t, strings.Contains(err.Error(), "not set"))
}

func TestSpadeReturnsNoError(t *testing.T) {
	fakeClient := &spadefakes.FakeClient{}
	fakeClient.TrackEventReturns(nil)
	wrapper := &WrapperClient{
		client: fakeClient,
	}

	props := map[string]interface{}{
		"testkey1": "testkey1",
		"testkey2": 2,
	}
	err := wrapper.sendEvent("test", props)
	require.Nil(t, err)
	require.Equal(t, 1, fakeClient.TrackEventCallCount())
	_, event, properties := fakeClient.TrackEventArgsForCall(0)
	require.Equal(t, "test", event)
	require.Equal(t, props, properties)
}

func TestSpadeReturnsError(t *testing.T) {
	fakeClient := &spadefakes.FakeClient{}
	fakeClient.TrackEventReturns(fmt.Errorf("error"))
	wrapper := &WrapperClient{
		client: fakeClient,
	}

	err := wrapper.sendEvent("test", "test")
	require.NotNil(t, err)
	require.Equal(t, 1, fakeClient.TrackEventCallCount())
}

func TestSendStreamStartNoErrors(t *testing.T) {
	fakeClient := &spadefakes.FakeClient{}
	fakeClient.TrackEventReturns(nil)
	wrapper := &WrapperClient{
		client: fakeClient,
	}

	requiredStreamStart := TranscodeStreamStart{
		Channel:         "test_channel",
		ChannelId:       1234,
		CustomerId:      "test",
		ContentId:       "test",
		TranscodeType:   "x264",
		StreamId:        7646,
		StartTime:       12345,
		ArchivesEnabled: true,
		IngestSessionId: "session_id",
	}

	wrapper.SendStreamStartEvent("test_channel", "test", "test", "session_id", "x264", 1234, 12345, 7646, true)
	require.Equal(t, 1, fakeClient.TrackEventCallCount())
	_, event, properties := fakeClient.TrackEventArgsForCall(0)
	require.Equal(t, "transcode_stream_start", event)
	require.Equal(t, requiredStreamStart, properties)
}
