// Package tenfoot integrates gotranscoder with the Tenfoot eventbus client
//
// The client itself is vendored from the video/tenfoot repository.
//
package tenfoot

import (
	"time"

	"context"
	"github.com/Sirupsen/logrus"
	"github.com/pkg/errors"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
	tenfoot_eventbus "code.justin.tv/video/tenfoot/eventbus"
	tenfoot_client "code.justin.tv/video/tenfoot/eventbus/client"
)

var (
	// N.B.: We replace this with a mock during testing.
	clientFactory = tenfoot_client.New
)

// Settings contains configurable options that affect the behavior of the Tenfoot eventbus client.
type Settings struct {
	SendTimeout     time.Duration
	ShutdownTimeout time.Duration
	LogLevel        string
	SockBasePath    string
}

// A Glue manages a Tenfoot evenbus client, the service discovery mechanism, and message generation.  All functions must
// be safe to call on a nil receiver.
type Glue interface {
	// Stop waits for the Tenfoot eventbus client to gracefully exit.  If it has not done so within a configurable
	// interval, it is ungracefully stopped.  You should cancel the `context.Context` you passed to `New` before calling
	// `Wait`.
	Wait() error

	// PostSegment generates a message describing a newly-available segment and sends it to all connected Tenfoot instances.
	// The segment must have been written to disk (i.e. be ready for a recipient of that message to open and read) before
	// PostSegment is called.
	PostSegment(seg *avdata.Segment) error

	// PostPlaylist generates a message describing a newly-available playlist and sends it to all connected Tenfoot
	// instances.
	PostPlaylist(playlist m3u8.Playlist, quality, playlistPath, playlistType string, adBreaks []*hlsext.AdBreakRequest, futureSegments []*hls.Segment, initSegmentUri string) error
}

type glue struct {
	l *logrus.Logger

	settings *Settings
	metadata *TranscodeMetadata
	client   tenfoot_client.Client
}

var _ Glue = (*glue)(nil)

// Noop returns a Glue that does nothing.
func Noop() Glue {
	return (*glue)(nil)
}

// New creates and starts a Tenfoot eventbus client with the given settings.  The client will run until the context is
// canceled.
func New(ctx context.Context, settings *Settings, metadata *TranscodeMetadata) (Glue, error) {
	g := &glue{
		l:        logrus.New(),
		settings: settings,
		metadata: metadata,
	}

	parsedLogLevel, err := logrus.ParseLevel(settings.LogLevel)
	if err != nil {
		return nil, errors.Wrap(err, "failed to parse log level")
	}
	g.l.Level = parsedLogLevel

	sessionMessage, err := g.metadata.buildSessionMessage()
	if err != nil {
		return nil, errors.Wrap(err, "failed to build session message")
	}

	discoConfig := tenfoot_eventbus.DiscoveryConfig.WithSockBasePath(g.settings.SockBasePath)
	g.client, err = clientFactory(g.l, sessionMessage, discoConfig)
	if err != nil {
		return nil, errors.Wrap(err, "failed to create Tenfoot eventbus client")
	}

	if err := g.client.Start(ctx); err != nil {
		return nil, errors.Wrap(err, "failed to start Tenfoot eventbus client")
	}

	return g, nil
}

func (g *glue) Wait() error {
	if g != nil {
		// Pause to give the client time to shut down and close connections gracefully.
		ctx, cancel := context.WithTimeout(context.Background(), g.settings.ShutdownTimeout)
		defer cancel()
		if err := g.client.Wait(ctx); err != nil {
			return errors.Wrap(err, "failed to shut down Tenfoot client gracefully")
		}
	}

	return nil
}

func (g *glue) PostSegment(seg *avdata.Segment) error {
	if g != nil {
		msg, err := g.metadata.buildSegmentMessage(seg)
		if err != nil {
			return errors.Wrap(err, "failed to build Tenfoot segment message")
		}

		ctx, cancel := context.WithTimeout(context.Background(), g.settings.SendTimeout)
		defer cancel()
		if err := g.client.SendMessage(ctx, msg); err != nil {
			return errors.Wrap(err, "failed to send Tenfoot segment message")
		}
	}

	return nil
}

func (g *glue) PostPlaylist(playlist m3u8.Playlist, quality, playlistPath, playlistType string, adBreaks []*hlsext.AdBreakRequest, futureSegments []*hls.Segment, initSegmentUri string) error {
	if g != nil {

		creation := time.Now()

		msg, err := g.metadata.buildPlaylistMessage(playlist, creation, quality, playlistPath, playlistType, adBreaks, initSegmentUri)
		if err != nil {
			return errors.Wrap(err, "failed to build Tenfoot playlist message")
		}

		ctx, cancel := context.WithTimeout(context.Background(), g.settings.SendTimeout)
		defer cancel()
		if err := g.client.SendMessage(ctx, msg); err != nil {
			return errors.Wrap(err, "failed to send Tenfoot playlist message")
		}

	}

	return nil
}
