package tenfoot

import (
	"fmt"
	"path"
	"sync"
	"testing"
	"time"

	"context"
	"github.com/Sirupsen/logrus"
	"github.com/WatchBeam/clock"
	"github.com/golang/protobuf/ptypes"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	eventbuslib_common "code.justin.tv/video/eventbuslib/common"
	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
	"code.justin.tv/video/protocols/tenfoot"
	tenfoot_client "code.justin.tv/video/tenfoot/eventbus/client"
)

type mockTenfootClientFactory struct {
	mu    sync.Mutex
	mocks []*mockTenfootClient

	newCb func(*mockTenfootClient) error
}

func (f *mockTenfootClientFactory) New(l *logrus.Logger, sessionMessage *tenfoot.TranscoderMessage, discoConfig *eventbuslib_common.LocalDiscoveryConfig) (tenfoot_client.Client, error) {
	m := &mockTenfootClient{}

	if f.newCb != nil {
		if err := f.newCb(m); err != nil {
			return nil, err
		}
	}

	f.mu.Lock()
	defer f.mu.Unlock()
	f.mocks = append(f.mocks, m)

	return m, nil
}

// @KK: Why are these unexported when AssertExpectations is?
func (f *mockTenfootClientFactory) mockQty() int {
	f.mu.Lock()
	defer f.mu.Unlock()
	return len(f.mocks)
}

func (f *mockTenfootClientFactory) mock(i int) *mockTenfootClient {
	f.mu.Lock()
	defer f.mu.Unlock()
	return f.mocks[i]
}

func (f *mockTenfootClientFactory) AssertExpectations(t *testing.T) bool {
	f.mu.Lock()
	defer f.mu.Unlock()

	ok := true
	for _, mock := range f.mocks {
		ok = ok && mock.AssertExpectations(t)
	}
	return ok
}

type mockTenfootClient struct {
	mock.Mock

	mu   sync.Mutex
	msgs []*tenfoot.TranscoderMessage
}

func (m *mockTenfootClient) Start(ctx context.Context) error {
	retval := m.Called()
	return retval.Error(0)
}

func (m *mockTenfootClient) Wait(ctx context.Context) error {
	retval := m.Called()
	return retval.Error(0)
}

func (m *mockTenfootClient) SendMessage(ctx context.Context, msg *tenfoot.TranscoderMessage) error {
	m.mu.Lock()
	defer m.mu.Unlock()
	m.msgs = append(m.msgs, msg)

	retval := m.Called()
	return retval.Error(0)
}

func (m *mockTenfootClient) assertNumberOfCalls(t *testing.T, counts map[string]int) bool {
	fns := []string{"Start", "Wait", "SendMessage"}

	ok := true

	for _, fn := range fns {
		expected, _ := counts[fn]
		ok = ok && m.AssertNumberOfCalls(t, fn, expected)
	}

	for k := range counts {
		found := false
		for _, fn := range fns {
			if fn == k {
				found = true
			}
		}
		assert.True(t, found, fmt.Sprintf("unexpected function name: %v", k))
	}

	return ok
}

func (m *mockTenfootClient) messages() []*tenfoot.TranscoderMessage {
	m.mu.Lock()
	defer m.mu.Unlock()

	return m.msgs
}

var _ tenfoot_client.Client = (*mockTenfootClient)(nil)

type TenfootGlueTestSuite struct {
	suite.Suite

	clk               clock.Clock
	mockClientFactory *mockTenfootClientFactory

	// Fixtures.
	settings *Settings
	metadata *TranscodeMetadata
}

func TestTenfootGlueTestSuite(t *testing.T) {
	suite.Run(t, new(TenfootGlueTestSuite))
}

func (suite *TenfootGlueTestSuite) SetupTest() {
	suite.clk = clock.DefaultClock{}

	suite.mockClientFactory = &mockTenfootClientFactory{}
	suite.mockClientFactory.newCb = func(m *mockTenfootClient) error {
		m.On("Start", mock.Anything).Return(nil)
		m.On("SendMessage", mock.Anything, mock.Anything).Return(nil)
		m.On("Wait").Return(nil)
		return nil
	}

	clientFactory = suite.mockClientFactory.New // N.B.: This is a global that New() uses.

	suite.settings = &Settings{
		SendTimeout:     250 * time.Millisecond,
		ShutdownTimeout: 500 * time.Millisecond,
		LogLevel:        "INFO",
		SockBasePath:    "/tmp/no-such-path",
	}
	suite.metadata = &TranscodeMetadata{
		StartTimestamp:        suite.clk.Now(),
		TargetSegmentDuration: 2 * time.Second,
		TranscodeProfile:      "Transcode2015Main",
		Qualities:             []string{"chunked", "audio_only"},
		Channel:               "foouser",
		UsherStreamID:         123,
		SessionID:             uuid.NewV4(),
		ResourcePath:          "foouser_123_456",
		OriginHostname:        "hls-28bca1",
		FileFormat:            hlsext.ExtendedPlaylist_TS,
	}
}

func (suite *TenfootGlueTestSuite) TearDownTest() {
}

// When we get a Glue from Noop(), all of its functions should simply succeed without doing anything.
func (suite *TenfootGlueTestSuite) TestNoopGlue() {
	t := suite.T()

	g := Noop()
	assert.Nil(t, g.Wait())
	assert.Nil(t, g.PostSegment(nil))
	assert.Nil(t, g.PostPlaylist(m3u8.Playlist{}, "", "", "", nil, nil, ""))
}

// Check assertions concerning the TranscoderMessage wrapper.  Note that this calls Validate() on the message, which
// also validates submessages.
func (suite *TenfootGlueTestSuite) validateAndCheckWrapper(outerMsg *tenfoot.TranscoderMessage, quality string) bool {
	t := suite.T()

	// Message must be valid and have version 0.
	v, err := outerMsg.Validate()
	if !(assert.Nil(t, err, "message failed validation") && assert.Equal(t, int64(0), v, "unexpected message version")) {
		return false
	}

	s := outerMsg.Stream

	ok := true
	ok = ok && assert.Equal(t, suite.metadata.SessionID.Bytes(), s.BroadcastId)
	ok = ok && assert.Equal(t, suite.metadata.UsherStreamID, s.UsherStreamId)
	ok = ok && assert.Equal(t, suite.metadata.Channel, s.Channel)
	ok = ok && assert.Equal(t, quality, s.Quality)
	ok = ok && assert.Equal(t,
		path.Join(suite.metadata.OriginHostname, suite.metadata.ResourcePath),
		s.OriginResourcePath)

	// TODO: Validate() checks that msg.Timestamp is set; can we be more strict here?

	return ok
}

func (suite *TenfootGlueTestSuite) TestNew() {
	t := suite.T()

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	g, err := New(ctx, suite.settings, suite.metadata)
	if !(assert.Nil(t, err) && assert.NotNil(t, g)) {
		return
	}

	assert.Equal(t, 1, suite.mockClientFactory.mockQty())
	m := suite.mockClientFactory.mock(0)
	m.assertNumberOfCalls(t, map[string]int{"Start": 1})
}

func (suite *TenfootGlueTestSuite) TestWait() {
	t := suite.T()

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	g, err := New(ctx, suite.settings, suite.metadata)
	if !(assert.Nil(t, err) && assert.NotNil(t, g)) {
		return
	}

	cancel()
	err = g.Wait()
	if !assert.Nil(t, err) {
		return
	}

	assert.Equal(t, 1, suite.mockClientFactory.mockQty())
	m := suite.mockClientFactory.mock(0)
	m.assertNumberOfCalls(t, map[string]int{"Start": 1, "Wait": 1})
}

func (suite *TenfootGlueTestSuite) TestBuildSessionMessage() {
	t := suite.T()

	outerMsg, err := suite.metadata.buildSessionMessage()
	if !assert.Nil(t, err) {
		return
	}

	if !suite.validateAndCheckWrapper(outerMsg, "") {
		return
	}

	// Validate the TranscodeSession submessage.
	startProto, err := ptypes.TimestampProto(suite.metadata.StartTimestamp)
	if err != nil {
		panic(err)
	}
	msg := outerMsg.Msg.(*tenfoot.TranscoderMessage_TranscodeSession)
	assert.Equal(t, suite.metadata.TranscodeProfile, msg.TranscodeSession.Profile)
	assert.Equal(t, suite.metadata.Qualities, msg.TranscodeSession.Qualities)
	assert.Equal(t, startProto, msg.TranscodeSession.Start)
	assert.Equal(t, ptypes.DurationProto(suite.metadata.TargetSegmentDuration),
		msg.TranscodeSession.TargetSegmentDuration)
}

func (suite *TenfootGlueTestSuite) TestPostSegment() {
	t := suite.T()

	avSeg := &avdata.Segment{
		Label:         "chunked",
		SegmentNumber: 42,
		SegmentName:   "index-0042-abcd.ts",
		Duration:      2000.0,
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	g, err := New(ctx, suite.settings, suite.metadata)
	if !(assert.Nil(t, err) && assert.NotNil(t, g)) {
		return
	}

	err = g.PostSegment(avSeg)
	if !assert.Nil(t, nil) {
		return
	}

	assert.Equal(t, 1, suite.mockClientFactory.mockQty())
	m := suite.mockClientFactory.mock(0)
	m.assertNumberOfCalls(t, map[string]int{"Start": 1, "SendMessage": 1})

	outerMsgs := m.messages()
	assert.Equal(t, 1, len(outerMsgs))
	if !suite.validateAndCheckWrapper(outerMsgs[0], avSeg.Label) {
		return
	}

	// Equivalent to 'path.Join(suite.metadata.ResourcePath, avSeg.Label, avSeg.SegmentName)'
	expectedPath := "foouser_123_456/chunked/index-0042-abcd.ts"

	msg := outerMsgs[0].Msg.(*tenfoot.TranscoderMessage_Segment)
	assert.Equal(t, expectedPath, msg.Segment.Path) //Hardwired!
	assert.Equal(t, avSeg.SegmentName, msg.Segment.Segment.Uri)
	assert.Equal(t, avSeg.SegmentNumber, msg.Segment.Segment.SequenceNumber)
	assert.Equal(t, ptypes.DurationProto(2*time.Second), msg.Segment.Segment.Duration) // Hardwired!
	assert.False(t, msg.Segment.Segment.Discontinuity)
}

func (suite *TenfootGlueTestSuite) TestPostPlaylist() {
	t := suite.T()

	playlist := m3u8.Playlist{
		Chunks:         []m3u8.Chunk{},
		StreamDuration: 1000.0,
		StreamOffset:   1000.0,
		TargetDuration: 3,
		IsFinal:        false,
		WindowSize:     6,
		MediaSequence:  2,
		PlaylistType:   m3u8.PlaylistTypeLive,
	}
	quality := "chunked"
	playlistPath := ""
	playlistType := "live"
	adBreaks := []*hlsext.AdBreakRequest{}
	futureSegments := []*hls.Segment{}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	g, err := New(ctx, suite.settings, suite.metadata)
	if !(assert.Nil(t, err) && assert.NotNil(t, g)) {
		return
	}

	err = g.PostPlaylist(playlist, quality, playlistPath, playlistType, adBreaks, futureSegments, "")
	if !assert.Nil(t, nil) {
		return
	}

	assert.Equal(t, 1, suite.mockClientFactory.mockQty())
	m := suite.mockClientFactory.mock(0)
	m.assertNumberOfCalls(t, map[string]int{"Start": 1, "SendMessage": 1})

	outerMsgs := m.messages()
	assert.Equal(t, 1, len(outerMsgs))
	if !suite.validateAndCheckWrapper(outerMsgs[0], quality) {
		return
	}

	// Check that message is of expected type.
	_ = outerMsgs[0].Msg.(*tenfoot.TranscoderMessage_Playlist).Playlist
}

func (suite *TenfootGlueTestSuite) TestPostPlaylistFmp4() {
	t := suite.T()

	playlist := m3u8.Playlist{
		Chunks:         []m3u8.Chunk{},
		StreamDuration: 1000.0,
		StreamOffset:   1000.0,
		TargetDuration: 3,
		IsFinal:        false,
		WindowSize:     6,
		MediaSequence:  2,
		PlaylistType:   m3u8.PlaylistTypeLive,
	}
	quality := "chunked"
	playlistPath := ""
	playlistType := "live"
	adBreaks := []*hlsext.AdBreakRequest{}
	futureSegments := []*hls.Segment{}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()

	suite.metadata.FileFormat = hlsext.ExtendedPlaylist_MP4
	g, err := New(ctx, suite.settings, suite.metadata)
	if !(assert.Nil(t, err) && assert.NotNil(t, g)) {
		return
	}

	err = g.PostPlaylist(playlist, quality, playlistPath, playlistType, adBreaks, futureSegments, "moov.mp4")
	if !assert.Nil(t, nil) {
		return
	}

	assert.Equal(t, 1, suite.mockClientFactory.mockQty())
	m := suite.mockClientFactory.mock(0)
	m.assertNumberOfCalls(t, map[string]int{"Start": 1, "SendMessage": 1})

	outerMsgs := m.messages()
	assert.Equal(t, 1, len(outerMsgs))
	if !suite.validateAndCheckWrapper(outerMsgs[0], quality) {
		return
	}

	// Check that message is of expected type.
	pl := outerMsgs[0].Msg.(*tenfoot.TranscoderMessage_Playlist).Playlist
	assert.Equal(t, tenfoot.NewPlaylist_M3U8_LIVE, pl.Type)
	assert.Equal(t, hlsext.ExtendedPlaylist_MP4, pl.Playlist.FileFormat)
	assert.Equal(t, "moov.mp4", pl.Playlist.InitSegmentUri)
}
