package tenfoot

import (
	"context"
	"log"
	"net/http"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/gotranscoder/pkg/origin"
	"code.justin.tv/video/origin/rpc/originctl"
	"code.justin.tv/video/origin/rpc/originfoot"
	"code.justin.tv/video/protocols/hls"
	"code.justin.tv/video/protocols/hlsext"
)

// Time to wait before timing out requests to SetPlaylist on originfoot.
const originSetPlaylistTimeout = 3 * time.Second

// OriginFoot is a shim that lets gotranscoder send tenfoot-style playlist data
// to Origin. It always performs Wait, PostSegment, and PostPlaylist requests by
// sending them to Tenfoot first. If that succeeds, it tries sending to Origin.
// If Origin errors, it merely logs that error. This makes OriginFoot safe to
// use, even if Origin is erroring, but this should be changed when Origin is
// stable.
type OriginFoot struct {
	base Glue

	client   originfoot.OriginTenfootShim
	session  *originctl.Session
	metadata *TranscodeMetadata

	timeout time.Duration
}

// NewOriginFoot constructs tenfoot-style handler for origin
func NewOriginFoot(base Glue, cfg origin.Config, session *originctl.Session, metadata *TranscodeMetadata) *OriginFoot {
	return &OriginFoot{
		base:     base,
		client:   originfoot.NewOriginTenfootShimProtobufClient(cfg.Endpoint, http.DefaultClient),
		session:  session,
		metadata: metadata,
		timeout:  originSetPlaylistTimeout,
	}
}

// Wait TODO: document me
func (o *OriginFoot) Wait() error { return o.base.Wait() }

// PostSegment just calls PostSegment on the base glue. The Origin service does
// not receive segments like Tenfoot does; it learns about them directly from
// the transcoder.
func (o *OriginFoot) PostSegment(seg *avdata.Segment) error { return o.base.PostSegment(seg) }

// PostPlaylist sends a Tenfoot playlist to the base glue, and then to the
// Origin service.
func (o *OriginFoot) PostPlaylist(playlist m3u8.Playlist, quality, playlistPath, playlistType string, adBreaks []*hlsext.AdBreakRequest, futureSegments []*hls.Segment, initSegmentUri string) error {
	err := o.base.PostPlaylist(playlist, quality, playlistPath, playlistType, adBreaks, futureSegments, initSegmentUri)
	if err != nil {
		log.Printf("[ORIGINFOOT] base error, not setting playlist: %s", err)
		return err
	}

	log.Printf("[ORIGINFOOT] setting playlist for quality=%v, path=%v, type=%v", quality, playlistPath, playlistType)
	creation := time.Now()
	tenfootPlaylist, err := o.metadata.buildPlaylist(playlist, creation, quality, playlistPath, playlistType, adBreaks, futureSegments, initSegmentUri)
	if err != nil {
		log.Printf("[ORIGINFOOT] failed to build Tenfoot playlist message: %s", err)
		return nil
	}

	ctx, cancel := context.WithTimeout(context.TODO(), o.timeout)
	defer cancel()
	req := &originfoot.SetPlaylistRequest{
		Session:  o.session,
		Playlist: tenfootPlaylist,
	}
	_, err = o.client.SetPlaylist(ctx, req)
	if err != nil {
		log.Printf("[ORIGINFOOT] failed to set playlist: %s", err)
		return nil
	}
	log.Printf("[ORIGINFOOT] playlist set successfully")
	return nil
}
