/*
Package twitchtranscoder parses Twitch transcoder output - looks up for json values of debug information
*/
package twitchtranscoder

import (
	"encoding/json"
	"strconv"
	"strings"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
)

// Markers to lookup in libavtwitch's output to help in parsing
const (
	debugDataMarker         string = "DEBUG:"
	segmentDataMarker       string = "segment_name"
	fmp4InitSegmentMarker   string = "init_segment"
	filenameDataMarker      string = "filename_prefix"
	thumbnailDataMarker     string = "thumb"
	streamlogDataMarker     string = "STREAMLOG"
	transcoderlogDataMarker string = "TWITCHTRANSCODERLOG"
	SpritesDataMarker       string = "SPRITES_MARKER"
)

// Extract the value from a twitch transcoder log line based on a lookup constant
// example:
// extracts the value 60 out for ---time_scale
func Extract(line string, key string) string {
	startPoint := strings.Index(line, key) + len(key)
	datum := strings.Split(line[startPoint:], " ")

	if len(datum) > 0 {
		// cleanup trailing commas
		if strings.HasSuffix(datum[0], ",") {
			return datum[0][:len(datum[0])-1]
		}

		return datum[0]
	}

	return ""
}

// ParseDebug looks for debug information in the TwitchTranscoder output lines
// to avoid the heavy string parsing.
func ParseDebug(line string) map[string]interface{} {
	var debugData map[string]interface{}

	if strings.Contains(line, debugDataMarker) {
		debugData = make(map[string]interface{})

		for key, value := range avdata.DebugMarkers {

			if strings.Contains(line, value) {
				data := Extract(line, value)
				switch key {
				case "Bitrate":
					value, _ := strconv.ParseFloat(data[:len(data)-4], 64)
					debugData[key] = value
				case "Width", "Height":
					value, _ := strconv.ParseFloat(data, 64)
					debugData[key] = value
				default:
					debugData[key] = data
				}

			}
		}
	}

	if len(debugData) < 1 {
		return nil
	}

	return debugData
}

// ParseLine gets out json metadata from the output of twitch transcoder
func ParseLine(data []byte) (interface{}, error) {
	// TODO: plan is to have libavtwitch tag it's ouput with labels and use that
	// for lookups. Until then, just look for the initial character to be an open
	// brace and attenpt to parse as json. This avoids parsing every line.
	if strings.HasPrefix(string(data[0:1]), "{") {
		line := string(data)

		if strings.Contains(line, thumbnailDataMarker) {
			var thumb avdata.Thumbnail
			err := json.Unmarshal(data, &thumb)
			return thumb, err
		}

		if strings.Contains(line, segmentDataMarker) {
			var segment avdata.Segment
			err := json.Unmarshal(data, &segment)
			return segment, err
		}

		if strings.Contains(line, fmp4InitSegmentMarker) {
			var initSegment avdata.Fmp4InitSegment
			err := json.Unmarshal(data, &initSegment)
			return initSegment, err
		}

		if strings.Contains(line, filenameDataMarker) {
			var codec avdata.Codec
			err := json.Unmarshal(data, &codec)
			return codec, err
		}

		if strings.Contains(line, transcoderlogDataMarker) {
			var twitchTranscoderLog avdata.TwitchTranscoderLog
			err := json.Unmarshal(data, &twitchTranscoderLog)
			return twitchTranscoderLog, err
		}

		if strings.Contains(line, streamlogDataMarker) {
			var streamlog avdata.StreamLog
			err := json.Unmarshal(data, &streamlog)
			return streamlog, err
		}

		if strings.Contains(line, SpritesDataMarker) {
			var spritesheet avdata.Sprite
			err := json.Unmarshal(data, &spritesheet)
			return spritesheet, err
		}

	}

	return nil, nil
}
