package twitchtranscoder

import (
	"testing"

	"code.justin.tv/video/gotranscoder/pkg/avdata"

	"github.com/stretchr/testify/assert"
	"strings"
)

var (
	debugMessage1 = `libavtwitch: [ avtwitch_videoutils.cpp:234 ] DEBUG: ---video_format: 5`
	debugMessage2 = `libavtwitch: [ avtwitch_videoutils.cpp:227 ] DEBUG: ---time_scale: 120`
	debugMessage3 = `libavtwitch: [ avtwitch_encode.cpp:227 ] DEBUG: Video Encoder initialized (0x7fe98b809600). width=1280, height=720, bitrate=1562kbps, preset=veryfast, tune=null, profile=Baseline, level=31, codec=avc1.42C01F`

	regularOutput = `libavtwitch: [ avtwitch_segment.cpp:51 ] DEBUG: Starting segment 0 at PTS 2314470.997732 (RAI: true)`

	thumbnailJSON   = `{"label":"thumb","luminosity":0.0000000000000000,"size":245806,"thumb_name_latest":"/tmp/archives/im_test/thumb/thumb.jpg","timestamp":2601049}`
	audioCodecJSON  = `{"audio_codec":"mp4a.40.2","channels":2,"filename_prefix":"index","label":"chunked","sample_rate":44100,"segmentable":true}`
	videoCodecJSON  = `{"compatibility":192,"filename_prefix":"index"}`
	segmentJSON     = `{"VideoEncoderMetrics":{"bitrate":372000,"fps":30.000000000000000,"frame_cnt":83,"frame_out_of_sync_cnt":0,"preset":"veryfast","profile":"main"},"VideoQualityMetrics":{"PSNR_U":41.877908173192658,"PSNR_V":41.889948118172434,"PSNR_Y":40.470220213797923,"PSNR_avg":40.890614580773821,"SSIM":0.98385351825903811,"frame_cnt":60},"bandwidth":589944,"duration":4000,"frame_count":120,"idr_count":1,"label":"low","latency":7785.1760253906250,"max_idr_delta":3834,"psnr":null,"random_access_point":true,"segment_name":"index-0000000002-Gxdo.ts","segment_number":2,"segment_overhead":49188,"segment_overhead_percent":16.675481062609332,"segment_size":294972,"ssim":null,"timestamp":282033}`
	ttlogJSON       = `{"TWITCHTRANSCODERLOG":{"type":"warning","topic":"insert_silence","message":"encoder_type=quicksync,from=60000,to=60023,label=360p30,insert_reason=empty_segment"}}`
	invalidJSON     = `{"test" : 123}`
	initSegmentJSON = `{"label":"720p60","init_segment":"moov.mp4"}`
)

func TestDebugParser(t *testing.T) {
	assert := assert.New(t)

	out := ParseDebug(debugMessage1)
	assert.NotEmpty(out["VideoFormat"])
	assert.Equal("5", out["VideoFormat"])

	out = ParseDebug(debugMessage2)
	assert.NotEmpty(out["TimeScale"])
	assert.Equal("120", out["TimeScale"])

	out = ParseDebug(debugMessage3)
	assert.NotEmpty(out["Width"])
	assert.NotEmpty(out["Height"])
	assert.Equal(1280.0, out["Width"])
	assert.Equal(720.0, out["Height"])
	assert.Equal(1562.0, out["Bitrate"])

	out = ParseDebug(regularOutput)
	assert.Nil(out)

	out = ParseDebug(segmentJSON)
	assert.Nil(out)

	out = ParseDebug(ttlogJSON)
	assert.Nil(out)
}

func TestJsonParser(t *testing.T) {
	assert := assert.New(t)

	var thumb avdata.Thumbnail
	out, err := ParseLine([]byte(thumbnailJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(thumb, out)

	var codec avdata.Codec
	out, err = ParseLine([]byte(audioCodecJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(codec, out)

	out, err = ParseLine([]byte(videoCodecJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(codec, out)

	var segment avdata.Segment
	out, err = ParseLine([]byte(segmentJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(segment, out)

	var ttlog avdata.TwitchTranscoderLog
	out, err = ParseLine([]byte(ttlogJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(ttlog, out)

	outJson, ok := out.(avdata.TwitchTranscoderLog)
	assert.True(ok)
	assert.True(strings.Contains(outJson.TTLData.Message, "empty_segment"))
	assert.Equal(outJson.TTLData.Type, "warning")
	assert.Equal(outJson.TTLData.Topic, "insert_silence")

	var initSegment avdata.Fmp4InitSegment
	out, err = ParseLine([]byte(initSegmentJSON))
	assert.Nil(err)
	assert.NotNil(out)
	assert.IsType(initSegment, out)

	out, err = ParseLine([]byte(invalidJSON))
	assert.Nil(err)
	assert.Nil(out)
}
