package twitchtranscoder

import (
	"testing"

	"strings"

	"io/ioutil"

	"time"

	"github.com/stretchr/testify/assert"
)

var twitchTranscoderProbePath = "./pkg/twitchtranscoder/_build_artifacts/mock_twitch_probe"

var mockCompileProbePath = "../../cmd/mocktwitchprobe"

func TestParsingEmptyProbeResult(t *testing.T) {
	assert := assert.New(t)
	out, err := ParseProbeOutput("")
	assert.EqualValues(true, err)
	assert.Equal(0, int(out.ProbeAV.Video.Height))
}

func TestParsingBadJsonResponse(t *testing.T) {
	assert := assert.New(t)
	out, err := ParseProbeOutput("twitch_probe_result:{\"segment\": 0, \"total_segments\": 1")
	assert.EqualValues(true, err)
	assert.Equal(0, int(out.ProbeAV.Video.Height))
}

func TestParsingInvalidHeight(t *testing.T) {
	assert := assert.New(t)
	out, err := ParseProbeOutput("{\"twitch_probe_result\":{\"Video\":{\"height\":0}}}")
	assert.EqualValues(true, err)
	assert.Equal(0, int(out.ProbeAV.Video.Height))
}

func TestParsingErrorFromProbe(t *testing.T) {
	assert := assert.New(t)
	_, err := ParseProbeOutput("{\"twitch_probe_result\":{\"Video\":{\"Height\":10,\"Width\":20},\"error\":\"error seen\"}}")
	assert.EqualValues(true, err)
}

func TestParsingNoError(t *testing.T) {
	assert := assert.New(t)
	out, err := ParseProbeOutput("{\"twitch_probe_result\":{\"Video\":{\"max_idr_interval\":2000,\"Height\":15,\"Width\":20,\"Fps\":30.0,\"Bitrate\":3000}}}")
	assert.EqualValues(false, err)
	assert.EqualValues(15, int(out.ProbeAV.Video.Height))
	assert.EqualValues(20, int(out.ProbeAV.Video.Width))
	assert.EqualValues(30, int(out.ProbeAV.Video.Fps))
	assert.EqualValues(3000, int(out.ProbeAV.Video.Bitrate))
	assert.EqualValues(2000, int(out.ProbeAV.Video.MaxIdrInterval))
}

func TestParseOutput(t *testing.T) {
	assert := assert.New(t)
	out := parseOutput("")
	assert.EqualValues("", out)
}

func TestParseOutputProbeFailed(t *testing.T) {
	assert := assert.New(t)
	out := parseOutput("the probe failed")
	assert.EqualValues("", out)
}

func TestProbeValueExtraction(t *testing.T) {
	assert := assert.New(t)
	data, err := ioutil.ReadFile("testdata/probe_result.txt")
	if err != nil {
		assert.Fail("Error in reading data from file")
	}

	out := parseOutput(string(data))
	assert.NotEqual("", out)
	assert.True(len(out) > 0)
	assert.True(strings.Contains(out, "twitch_probe_result"))
}

func TestRunProbeCommand(t *testing.T) {
	//Mock is build as the process of building the twitch transcoder mock
	//Use the channel name to basically switch the type of output the mock probe prints
	SetProbeTimeoutDuration(5 * time.Second)
	assert := assert.New(t)
	output := RunTwitchProbe("probe_successful", twitchTranscoderProbePath)
	assert.True(strings.Contains(output, "twitch_probe_result"))
}

//Return nothing when the command does not complete withing the stipulated amount of time
func TestProbeTimeout(t *testing.T) {
	SetProbeTimeoutDuration(1 * time.Second)
	assert := assert.New(t)
	output := RunTwitchProbe("probe_hung", twitchTranscoderProbePath)
	assert.EqualValues("", output)

}
