package usher

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

var liveUsher = Settings{
	BaseURL: "http://usher.twitch.tv/",
}

var mockHTTP = Settings{
	BaseURL: "http://127.0.0.1:1234/",
}

var stateless = Settings{
	BaseURL:   "http://127.0.0.1:1234/",
	Stateless: true,
}

var testChannel = `twitchmedia2`

func TestPost(t *testing.T) {
	assert := assert.New(t)

	testData := `{"T":"T"}`
	data, err := liveUsher.Post(`/echo.json`, testData)
	assert.Nil(err)
	assert.NotNil(data)
}

func TestGet(t *testing.T) {
	assert := assert.New(t)
	data, err := liveUsher.Get(`running`, "")
	assert.Nil(err)
	assert.NotNil(data)
}

func TestPostStateless(t *testing.T) {
	assert := assert.New(t)

	testData := `{"T":"T"}`
	data, err := stateless.Post(`/echo.json`, testData)
	assert.Nil(err)
	assert.Nil(data)
}

func TestGetShowChannel(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.GetShowChannel(testChannel)
	assert.Nil(err)
	assert.NotNil(response)
	assert.Equal(response.Channel, testChannel)
}

func TestGetShowChannelNonExistent(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.GetShowChannel(`A`)
	assert.NotNil(err)
	assert.Nil(response)
}

func TestGetHLSTranscode(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.GetShowChannel(testChannel)
	assert.Nil(err)
	assert.NotNil(response)

	response2, err := liveUsher.GetHlsTranscode(response.ID)
	assert.Nil(err)

	assert.NotNil(response2)
	assert.Equal(response2.Channel, response.Channel)
}

func TestGetHLSTranscodeNonExistent(t *testing.T) {
	t.Skip("Disabled test until ING-1793 is fixed")
	assert := assert.New(t)
	response, err := liveUsher.GetHlsTranscode(0)
	assert.NotNil(err)
	assert.Nil(response)
}

func TestMarkStreamReady(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.GetShowChannel(testChannel)
	assert.Nil(err)
	assert.NotNil(response)

	err = liveUsher.MarkHlsStreamReady(response.Destination)
	assert.Nil(err)
}

func TestStreamInfo(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.StreamInfo(testChannel)
	assert.Nil(err)
	assert.NotNil(response)
	assert.NotEmpty(response)
	assert.Equal(response[0].Channel, testChannel)
}

func TestKillTranscode(t *testing.T) {
	t.Skip("Disabled test until ING-1793 is fixed")
	assert := assert.New(t)
	// wish we could test here against a real transcode but this is live data
	err := liveUsher.KillHlsTranscode(0)
	assert.NotNil(err)
	assert.Contains(err.Error(), "Status[404]")
}

func TestGetDBOption(t *testing.T) {
	assert := assert.New(t)
	response, err := liveUsher.GetDbOption("hls_transcode_update_interval")
	assert.Nil(err)
	assert.NotNil(response)
	assert.Equal(25.0, response.Value)
}

func TestGetStreamCodec(t *testing.T) {
	assert := assert.New(t)
	code, err := liveUsher.GetStreamCodec(testChannel)
	assert.Nil(err)
	assert.NotNil(code)
	assert.NotNil(code.VideoCodec)
}

func TestUpdateHlsEntry(t *testing.T) {
	assert := assert.New(t)
	assert.Error(liveUsher.UpdateHlsEntry(0, nil))
}

func TestUpdateHlsTranscodeEntry(t *testing.T) {
	assert := assert.New(t)
	var test HlsTranscode
	test.ID = 0
	// should have an URL
	assert.Error(liveUsher.UpdateHlsTranscode(0, &test))
}

func TestUpdateStreamProperties(t *testing.T) {
	assert := assert.New(t)
	assert.Error(liveUsher.UpdateStreamProperties(0, nil))
}

func TestSendRabbitJob(t *testing.T) {
	assert := assert.New(t)
	err := liveUsher.SendRabbitJob(`testing.key`, 1234)
	assert.Nil(err)
}

func TestUpdateHlsEntryStateless(t *testing.T) {
	assert := assert.New(t)
	assert.NoError(stateless.UpdateHlsEntry(0, "test"))
}

func TestUpdateHlsTranscodeEntryStateless(t *testing.T) {
	assert := assert.New(t)
	var test HlsTranscode
	test.ID = 0
	test.Channel = "test"
	assert.NoError(stateless.UpdateHlsTranscode(0, &test))
}

func TestUpdateStreamPropertiesStateless(t *testing.T) {
	assert := assert.New(t)
	assert.NoError(stateless.UpdateStreamProperties(0, nil))
}

func TestSendRabbitJobStateless(t *testing.T) {
	assert := assert.New(t)
	assert.NoError(stateless.SendRabbitJob(`testing.key`, 1234))
}
