// Package vod takes segments from the transcoder and uploads them
// to our object store. Also handles uploading thumbnails and
// registering VODs. The 'Pusher' type is responsible for multiplexing
// segments into queues bassed on their label(quality), and handling codec and
// thumbnail updates. The 'Queue' type contains a buffered channel of
// segments to be processed. Each Queue contains a vod qualities' playlist
// and partial object state. Queues communicate encountered dicontinuities
// to each other through the Pusher's 'discontinuities' map. 'Quit()'ing the
// the pusher will close each Queue's buffered channel and wait for all workers
// to flush.
package vod

import (
	"bytes"
	"crypto/sha1"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"math/rand"
	"net/http"
	"os"
	"path"
	"sync"
	"time"

	"github.com/cenkalti/backoff"
	"github.com/pkg/errors"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/noisy"
	"code.justin.tv/video/gotranscoder/pkg/notify"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
	"code.justin.tv/video/gotranscoder/pkg/usher"
)

const (
	// VOD API routes
	registerURLFmt   = "http://%s/v1/vods/past_broadcast"
	finalizeURLFmt   = "http://%s/v1/vods/%d"
	thumbnailsURLFmt = "http://%s/v1/vods/%d/thumbnails"

	// Content-Type for thumbnail images
	thumbMIMEType = "image/jpeg"

	//Timeout for Origin ReadThumbnail requests
	defaultOriginHttpTimeout = 10 * time.Second

	spriteQueueLen = 128
)

const ( // Keys for statsd
	statsdVinylDialFailure    = "dial_failure"
	statsdEnabled             = "vod.enabled"
	statsdRegisterOk          = "vod.register.ok"
	statsdRegisterFail        = "vod.register.fail"
	statsdVodRegisterSkipped  = "vod.register.skipped"
	statsdVodRegisterShort    = "vod.register.short"
	statsdFinalizeOk          = "vod.finalize.ok"
	statsdFinalizeFail        = "vod.finalize.fail"
	statsdVinylAPI            = "vod.vinyl_api"
	statsdVinylAPIError       = "vod.vinyl_api.retry"
	statsdAPIHit              = "vod.api.hit"        // successful request to site api
	statsdAPIError            = "vod.api.retry"      // error in request to site api
	statsdNewThumb            = "vod.new.thumbnail"  // Dropped thumbnail
	statsdLostThumb           = "vod.lost.thumbnail" // Dropped thumbnail
	statsdLostSprite          = "vod.lost.sprite"
	statsdThumbnailsAPIError  = "vod.thumbnails.api.retry" // error when requesting to create thumbnails
	constStatsSNSNotification = "notification.fail.%s"
	statsdLostManifest        = "vod.lost.manifest"
	statsdNewManifest         = "vod.new.manifest"
)

var registerNotifyTimeout = 60 * time.Second

// PusherSettings is used to configure a new Pusher
type PusherSettings struct {
	// Name of the channel
	ChannelName string

	//Id of the channel
	ChannelID uint64

	// ID for this broadcast
	BroadcastID int

	// Unique identifier for this VOD
	VodKey string

	// Absolute path to directory holding transcoded video segments and thumbnails
	TranscodePath string

	// how many segments per quality do we queue before dropping
	QueueSize int

	// how many segments per quality do we queue for notifying the vodpusher for upload
	NotifyQueueSize int

	// How many thumbnails to save for this VOD
	NumThumbnails int

	// Number of segments per playlist chunk
	ChunkFactor int

	// What to name playlist files
	PlaylistFileName string

	// Host of Vod Thumbnails API used to create vod's thumbnails
	VinylHost string

	// Usher object used to make calls to Usher
	Usher usher.Settings

	// Bucket to save our segments to
	Bucket string

	// Prefix for vod name
	Prefix string

	// Our s3 connection
	Conn Connection

	// soft-delete vod
	SoftDeleteVod bool

	// Minimum duration in seconds of VOD to be registered
	MinVodDurationSeconds int

	// Origin Server Session URL
	HlsUrlBase string

	OriginTimeout time.Duration

	// Disable Vinyl updates
	DisableVinyl bool

	Notifier notify.Notifier

	// for watch_party and watch_party_premiere
	BroadcasterSoftware string

	MaxIdrInterval int

	//Cloudfront distribution url for VOD playback , will be used in the vod manifest which will be stored in s3
	VodDistributionUrl string
}

// Pusher manages uploading video segments and metadata to our video-on-demand
// cluster. Segments, Thumbnails, and Codecs are supplied with 'Process()'. Pusher
// creates worker queues for each of these types and multiplexes inputs to their
// corresponding queue. Also handles VOD wide metadata and configurations, includeding
// regegistering the VOD when we start, and finalizing it when we've been signaled
// to Quit()
type Pusher struct {
	PusherSettings

	// Metadata about every VOD quality. Filled in by
	// the codec plugin callback
	formatsMux sync.Mutex
	formats    map[string]*format

	// VOD state
	vodID              int
	startedOn          time.Time
	registerTimer      *time.Timer
	outstandingWorkers sync.WaitGroup

	// One queue for each quality. Each queue processes
	// incoming segments and uploads them to s3
	queues map[string]*queue

	// A set of discontinuity sequence numbers encountered in any of the queues.
	discontinuitiesMux sync.Mutex
	discontinuities    map[int]struct{}

	thumbnailsMux sync.Mutex
	thumbnails    []thumbnail
	thumbQueue    chan *avdata.Thumbnail
	spriteQueue   chan *avdata.Sprite

	registerNotify chan bool

	// TODO: This should eventually be the same struct as TranscodeQualitySettings
	qualities map[string]Quality
}

// Quality represents a video quality for this stream
type Quality struct {
	Label              string
	DisplayName        string
	PlaylistPreference int
	Bitrate            int
	Fps                float64
}

// NewPusher constructs a new VOD pusher/uploader. All Pusher instances must
// be made with this function. "qualities" specifies which VOD qualities
// to handle. The 'Label' field on processed segments must match a quality's
// 'Label' field. Segments procesed with a "Label" not in "qualities"
// will be ignored
func NewPusher(ps PusherSettings, qualities map[string]Quality) *Pusher {
	log.Printf("[VOD] Registering new pusher \n")
	statsd.Inc(statsdEnabled, 1, 1)
	ps.VodKey = addPrefix(ps.VodKey, ps.Prefix) // s3 likes "random" prefixes

	if ps.OriginTimeout == 0 {
		ps.OriginTimeout = defaultOriginHttpTimeout
	}
	p := &Pusher{
		PusherSettings:  ps,
		formats:         make(map[string]*format, len(qualities)),
		queues:          make(map[string]*queue, len(qualities)),
		discontinuities: make(map[int]struct{}),
		thumbnails:      make([]thumbnail, 0, ps.NumThumbnails),
		thumbQueue:      make(chan *avdata.Thumbnail, 10),
		spriteQueue:     make(chan *avdata.Sprite, spriteQueueLen),
		registerNotify:  make(chan bool, 1),
		qualities:       qualities,
	}

	// Wait a bit before registering VOD
	p.outstandingWorkers.Add(1)
	p.registerTimer = time.NewTimer(registerNotifyTimeout)
	go p.registerVOD()

	// Initialize queue and format for each "Quality"
	for _, qual := range qualities {
		p.queues[qual.Label] = newQueue(p, qual.Label)
		p.formats[qual.Label] = &format{
			DisplayName:        qual.DisplayName,
			PlaylistPreference: qual.PlaylistPreference,
		}
	}

	log.Printf("[VOD] Initialized queues and formats - %+v \n, ", p.formats)
	p.outstandingWorkers.Add(2)
	go p.processThumbnails()
	go p.processSprites()

	return p
}

// Process is our external interface. This function is called as a
// callback when new transcoded segments or thumbnails are ready.
// Mutliplexes 'avdata.Segment', 'avdata.Thumbnail', and 'avdata.Codec'
// to the appropriate processing function
func (p *Pusher) Process(data interface{}, timeout <-chan struct{}) error {
	switch d := data.(type) {
	// pointer types
	case *avdata.Segment:
		return p.ProcessSegment(d)
	case *avdata.Thumbnail:
		return p.ProcessThumbnail(d)
	case *avdata.Codec:
		return p.ProcessCodec(d)
	case *avdata.Sprite:
		return p.processSprite(d)

	// Value types
	case avdata.Segment:
		return p.ProcessSegment(&d)
	case avdata.Thumbnail:
		return p.ProcessThumbnail(&d)
	case avdata.Codec:
		return p.ProcessCodec(&d)
	case avdata.Sprite:
		return p.processSprite(&d)
	default:
		return nil
	}
}

// ProcessSegment adds a segment to the appropriate processing queue based on
// its 'Label'(quality). Ignores segments whose label isn't a key in 'Pusher.queues'
func (p *Pusher) ProcessSegment(seg *avdata.Segment) error {
	log.Printf("[VOD] processing VOD segment %s-%s-%d\n", seg.Label, seg.SegmentName, seg.SegmentNumber)
	q, ok := p.queues[seg.Label]
	if !ok {
		err := fmt.Errorf("Unrecognized segment format for VOD: %s", seg.Label)
		log.Println(err)
		return err
	}

	//if startedOn is not set, set it to current time
	//startedOn will be set on the first segment processed.
	if p.startedOn.IsZero() {
		p.startedOn = time.Now()
		log.Printf("[VOD] setting StartedOn date to [%s]\n", p.startedOn)
		select {
		case p.registerNotify <- true:
		default:
		}
	}

	// Put the segment on the Pusher's queue
	select {
	case q.Notify <- seg:
	default:
		return fmt.Errorf("[VOD] Notify queue full")
	}

	return nil
}

// ProcessSprite adds a sprite to our sprite processing queue
func (p *Pusher) processSprite(sprite *avdata.Sprite) error {
	log.Printf("[VOD] processing sprite: %+v", sprite)
	select {
	case p.spriteQueue <- sprite:
	default:
		p.addStatHelper(statsdLostSprite, 1, 1.0)
		// Delete large unique files if there is a failure to push them on the queue
		if path.Ext(sprite.Path) != ".json" {
			err := os.Remove(sprite.Path)
			log.Printf("[VOD]failed to delete sprite asset '%s': %v", sprite.Path, err)
		}
		log.Println("[VOD] lost sprite")
		return fmt.Errorf("Lost sprite")
	}
	return nil
}

// ProcessThumbnail adds a thumbnail to our thumbnail processing queue
func (p *Pusher) ProcessThumbnail(thumb *avdata.Thumbnail) error {
	log.Printf("[VOD] processing thumbnail: %+v\n", thumb)
	select {
	case p.thumbQueue <- thumb:
	default:
		if len(p.thumbQueue) > 0 {
			<-p.thumbQueue //pop the oldest element in the queue
		}
		p.thumbQueue <- thumb
		p.addStatHelper(statsdLostThumb, 1, 1.0)
		log.Println("[VOD]  lost thumbnail")
		return fmt.Errorf("Lost thumbnail")
	}
	return nil
}

func (p *Pusher) processSprites() {
	defer p.outstandingWorkers.Done()

	for sprite := range p.spriteQueue {
		// Read the asset
		data, err := ioutil.ReadFile(sprite.Path)
		if err != nil {
			log.Printf("[VOD] pusher failed to read sprite asset '%s': %s", sprite.Path, err)
			statsd.Inc(statsdLostSprite, 1, 1.0)
			continue
		}

		// Try to write the asset to S3
		// TODO: reference a constant that specifies the sprite directory base
		// TODO: consider a non-retry Put path for .json as it gets updated periodically and
		//		retry store for the json file when finalizing the VOD
		_, filename := path.Split(sprite.Path)
		objName := fmt.Sprintf("%s/%s/%s", p.VodKey, "sprites", filename)
		mimeType := thumbMIMEType
		if path.Ext(filename) == ".json" {
			mimeType = "text/x-json"
		}

		if err := p.Conn.Put(p.Bucket, objName, data, mimeType); err != nil {
			log.Printf("[VOD] pusher couldn't upload sprite %s: %+v: %s", objName, sprite, err)
			statsd.Inc(statsdLostSprite, 1, 1.0)
		}

		// If the asset was committed to S3 it should be removed from disk
		if mimeType == thumbMIMEType {
			err = os.Remove(sprite.Path)
			if err != nil {
				log.Printf("[VOD] pusher failed to remove the asset '%s': %v", sprite.Path, err)
			}
		}
	}
}

// Thumbnail processing worker. Should only be called from 'NewPusher()'
// Thumbnails are chosen using reservoir sampling
func (p *Pusher) processThumbnails() {
	defer p.outstandingWorkers.Done()

	seenThumbnails := 0
	for thumb := range p.thumbQueue {
		index := rand.Intn(seenThumbnails + 1)
		log.Printf("[VOD] thumb seen index: %d\n", index)

		if index >= p.NumThumbnails {
			seenThumbnails++
			log.Printf("[VOD] pusher skipped thumbnail")
			continue
		} else if len(p.thumbnails) != p.NumThumbnails {
			// Set index to last element if reservoir isn't full
			index = len(p.thumbnails)
		}

		var data []byte
		var err error
		if p.HlsUrlBase != "" {
			// use Origin
			data, err = p.readThumbnailFromOrigin(thumb)
			if err != nil {
				// errors are logged inside the function
				continue
			}
		} else {
			data, err = p.readThumbnailFromFile(thumb)
			if err != nil {
				// errors are logged inside the function
				continue
			}
		}

		thumbPath := fmt.Sprintf("thumb/thumb%d.jpg", index)
		objName := path.Join(p.VodKey, thumbPath)
		if err := p.Conn.Put(p.Bucket, objName, data, thumbMIMEType); err != nil {
			log.Printf("[VOD] pusher couldn't upload thumbnail %s: %+v: %s\n", objName, thumb, err)
			statsd.Inc(statsdLostThumb, 1, 1.0)
			continue
		}

		t := thumbnail{
			Path:   thumbPath,
			Offset: time.Since(p.startedOn),
		}

		// If the reservoir isn't full, append
		p.thumbnailsMux.Lock()
		if len(p.thumbnails) != p.NumThumbnails {
			p.thumbnails = append(p.thumbnails, t)
		} else {
			p.thumbnails[index] = t
		}
		p.thumbnailsMux.Unlock()

		seenThumbnails++
		log.Printf("[VOD] pusher uploaded thumbnail: %+v\n", t)
		statsd.Inc(statsdNewThumb, 1, 1.0)

		if err := p.createThumbnails(); err != nil {
			log.Printf("[VOD] createThumbnails() failed: %s", err)
		}
	}
	log.Println("[VOD] exiting thumbnail worker")
}

// ProcessCodec updates codec information regarding this VOD. This is
// normally only called twice(video and audio codec)
func (p *Pusher) ProcessCodec(codec *avdata.Codec) error {
	p.formatsMux.Lock()
	defer p.formatsMux.Unlock()

	format, ok := p.formats[codec.Label]
	if !ok {
		msg := fmt.Sprintf("Codec for unrecognized label: %s", codec.Label)
		log.Println(msg)
		return errors.New(msg)
	}

	if codec.VideoCodec != "" {
		format.VideoCodec = codec.VideoCodec
		if codec.Width != 0 && codec.Height != 0 {
			format.Width = int(codec.Width)
			format.Height = int(codec.Height)
		}
	}
	if codec.AudioCodec != "" {
		format.AudioCodec = codec.AudioCodec
	}

	return nil
}

// Name returns this plugin's unique identifier
func (*Pusher) Name() string {
	return "VOD Pusher"
}

// Initialize is a no-op to satisfy the 'Plugin' interface
func (*Pusher) Initialize() {}

// Quit signals all processing go routines to quit and block until they have
// finished. Finalizes VOD after all workers are done. Further calls to
// 'Process' or 'Quit' will panic
func (p *Pusher) Quit() {
	close(p.thumbQueue)
	close(p.spriteQueue)
	for _, q := range p.queues {
		q.Quit()
	}

	// Register if we haven't yet
	if p.registerTimer.Stop() {
		close(p.registerNotify)
	}

	p.outstandingWorkers.Wait()
	p.finalizeVOD()
}

// Health returns true for a healthy vod pusher
func (p *Pusher) Health() bool {
	return p.queues["chunked"].Health()
}

// mark encountered dicontinuities from other qualities in this playlist
// return the amount of time lost from 'pl' due to marking
func (p *Pusher) markDiscontinuities(pl *Playlist) time.Duration {
	var lostDur float64
	p.discontinuitiesMux.Lock()
	for seqNum := range p.discontinuities {
		lostDur += pl.MarkDiscontinuous(seqNum)
	}
	p.discontinuitiesMux.Unlock()
	return time.Duration(lostDur*1e6) * time.Nanosecond
}

// Registers this vod and starts processing workers after we've
// been registered. Don't call this directly
func (p *Pusher) registerVOD() {
	defer p.outstandingWorkers.Done()

	select {
	case <-p.registerNotify:
		log.Println("[VOD] Registering vod on notification")
		p.registerVodHelper()
		p.registerTimer.Stop()
		err := p.Notifier.NotifyVodStart(fmt.Sprintf("%s/%s", p.Bucket, p.VodKey))
		if err != nil {
			log.Printf("[VOD] Failed to push SNS VodStart notification. Error: %v\n", err)
			statsd.Inc(fmt.Sprintf(constStatsSNSNotification, "vod_start"), 1, 1.0)
		} else {
			log.Println("[VOD] Successfully sent SNS VodStart notification")
		}
	case <-p.registerTimer.C:
		log.Printf("[VOD] No segments processed within %d seconds. vod will not be registered: %s", int(registerNotifyTimeout.Seconds()), p.ChannelName)
		statsd.Inc(statsdVodRegisterSkipped, 1, 1.0)
	}
}

//registerVodHelper is helper function to register vod with a deleted status
func (p *Pusher) registerVodHelper() {
	if p.DisableVinyl {
		log.Println("[VOD] Vinyl disabled. Skipping VOD registration")
		return
	}

	data, err := json.Marshal(&registerWrapper{
		PastBroadcast: registerVODProps{
			BroadcastID:         p.BroadcastID,
			BroadcastType:       "archive",
			BroadcasterSoftware: p.BroadcasterSoftware,
			RecordedOn:          p.startedOn,
			Status:              "recording",
			Channel:             p.ChannelName,
			OwnerID:             p.ChannelID,
			Formats:             p.getFormats(false),
			Offset:              0,
			Duration:            p.getDuration(),
			URI:                 p.VodKey,
			Origin:              "s3",
			Manifest:            p.PlaylistFileName,
			Deleted:             p.SoftDeleteVod,
		}})
	if err != nil {
		log.Printf("[VOD] '%s' will not be registered: %s", p.ChannelName, err)
		statsd.Inc(statsdRegisterFail, 1, 1.0)
		return
	}

	apiURL := fmt.Sprintf(registerURLFmt, p.VinylHost)
	log.Printf("[VOD] Registering vod: %s on :%s with DATA:{%+v}", p.ChannelName, apiURL, string(data))
	noisy.ESWrite("_", "Info", fmt.Sprintf("Vod Register Data: %s", string(data)))

	register := func() error {
		resp, postErr := jsonRequest(http.MethodPost, apiURL, data)
		if postErr != nil {
			log.Printf("[VOD] Error registering VOD: %v \n", postErr)
			reportVinylStatusCode(resp)
			if resp != nil {
				body, _ := ioutil.ReadAll(resp.Body)
				log.Printf("[VOD] Error response: %s \n", string(body))
				_ = resp.Body.Close()
			}
			statsd.Inc(statsdRegisterFail, 1, 1.0)
			return postErr
		}
		defer func() { _ = resp.Body.Close() }()
		// Read VOD id from json response
		type pastBroadcast struct {
			ID int `json:"id"`
		}

		var jsonResp struct {
			PastBroadcast pastBroadcast `json:"past_broadcast"`
		}

		if decErr := json.NewDecoder(resp.Body).Decode(&jsonResp); decErr != nil {
			log.Printf("[VOD] Error registering VOD with Vinyl: %s\n", decErr)
			statsd.Inc(statsdRegisterFail, 1, 1.0)
			return decErr
		}

		log.Printf("[VOD] Registered : %+v", jsonResp)
		// vodID shouldn't be accessed by other goroutines until
		// this function has finished (No more outstanding workers).
		p.vodID = jsonResp.PastBroadcast.ID
		return nil
	}

	exp := backoff.NewExponentialBackOff()
	exp.MaxElapsedTime = 0 //forever
	log.Printf("[VOD] Calling registry with exponential backoff\n")
	err = backoff.RetryNotify(register, exp, func(error, time.Duration) {
		statsd.Inc(statsdVinylAPIError, 1, 1.0)
	})
	if err != nil {
		log.Printf("[VOD] Vinyl API error: %s", err)
		return
	}

	log.Printf("VOD %d registered: %s with data: %s\n", p.vodID, p.VodKey, string(data))
	statsd.Inc(statsdRegisterOk, 1, 1.0)
	noisy.ESWrite("_", "Info", fmt.Sprintf("Vod ID: %d", p.vodID))
}

// FinalizeVOD finalizes this VOD in the site API.
// Must be called exactly once after all workers have finished
func (p *Pusher) finalizeVOD() {
	// This VOD hasn't been registered, so don't finalize
	if p.vodID == 0 && !p.DisableVinyl {
		log.Printf("[VOD] No VOD id to finalize - exit\n")
		return
	}

	log.Printf("[VOD] Finalizing VOD\n")

	// One final pass at updating playlists since all VOD qualities are
	// done and all possible discontinuities have been encountered
	wg := new(sync.WaitGroup)
	defer wg.Wait()
	for _, q := range p.queues {
		wg.Add(1)
		go func(q *queue) {
			err := backoff.Retry(q.uploadPlaylist, backoff.NewExponentialBackOff())
			if err != nil {
				log.Printf("[VOD] error uploading playlist: %s", err)
			}
			wg.Done()
		}(q)
	}
	err := p.Notifier.NotifyVodStop(fmt.Sprintf("%s/%s", p.Bucket, p.VodKey))
	if err != nil {
		statsd.Inc(fmt.Sprintf(constStatsSNSNotification, "vod_stop"), 1, 1.0)
		log.Printf("[VOD] Failed to push SNS VodStop notification. Error: %v\n", err)
	} else {
		log.Println("[VOD] Successfully sent SNS VodStop notification")
	}
	if p.DisableVinyl {
		log.Println("[VOD] Vinyl disabled. Skipping VOD finalization Vinyl update")
		return
	}

	err = p.externalFinalizeVod()
	if err != nil {
		log.Println(err.Error())
		statsd.Inc(statsdFinalizeFail, 1, 1.0)
		return
	}

	log.Printf("VOD %d finalized: %s\n", p.vodID, p.VodKey)
	statsd.Inc(statsdFinalizeOk, 1, 1.0)
}

// Isolate calls to external vod services in preparation to
// move the finalize vod functionality into a single Vinyl endpoint
func (p *Pusher) externalFinalizeVod() error {
	log.Printf("[VOD] Finalizing VOD on external vod services\n")
	finalizeData := &finalizeVODProps{
		Status:   "recorded",
		Formats:  p.getFormats(true),
		Duration: p.getDuration(),
		Deleted:  (p.getDuration().Seconds() < float64(p.PusherSettings.MinVodDurationSeconds)),
	}

	if finalizeData.Deleted {
		log.Printf("[VOD] Finalizing vod as soft-deleted. Vod-Duration: %f", finalizeData.Duration.Seconds())
		statsd.Inc(statsdVodRegisterShort, 1, 1.0)
	}

	data, err := json.Marshal(finalizeData)
	if err != nil {
		return fmt.Errorf("[VOD] Error finalizing VOD: %s\n", err)
	}

	noisy.ESWrite("_", "Info", fmt.Sprintf("Vod Finalize Data: %s", string(data)))

	apiURL := fmt.Sprintf(finalizeURLFmt, p.VinylHost, p.vodID)
	finalize := func() error {
		log.Printf("[VOD] PUT finalizing vod via %s", apiURL)
		resp, err2 := jsonRequest(http.MethodPut, apiURL, data)
		reportVinylStatusCode(resp)
		if err != nil {
			return err2
		}
		_ = resp.Body.Close()
		return nil
	}

	exp := backoff.NewExponentialBackOff()
	fErr := backoff.RetryNotify(finalize, exp, func(error, time.Duration) {
		statsd.Inc(statsdVinylAPIError, 1, 1.0)
	})
	if fErr != nil {
		log.Printf("[VOD] finalize error: %s", fErr)
	}

	log.Printf("[VOD] Queue Vod storyboard and mute jobs on vod rabbit")

	if !p.SoftDeleteVod {
		rErr := p.Usher.SendRabbitJob(constUsherAudioFilterKey, p.vodID)
		if rErr != nil {
			log.Printf("[VOD] failed to send audio filter job: %s", err)
		}
		rErr = p.Usher.SendRabbitJob(constUsherStoryboardKey, p.vodID)
		if rErr != nil {
			log.Printf("[VOD] failed to send storyboard job: %s", err)
		}
	}

	// Register the vod's thumbnails
	err = p.createThumbnails()
	if err != nil {
		return fmt.Errorf("[VOD] Error registering vod's thumbnails: %s\n", err)
	}

	log.Printf("[VOD] finalized")
	return nil
}

func (p *Pusher) createThumbnails() error {
	if p.DisableVinyl {
		log.Println("[VOD] Vinyl disabled. Skipping thumbnail creation")
		return nil
	}

	thumbnails := p.getThumbnails()
	if len(thumbnails) == 0 || p.vodID == 0 {
		return nil
	}

	thumbnailsData, err := json.Marshal(map[string]interface{}{
		"thumbnails": thumbnails,
	})
	if err != nil {
		return errors.New("Error marshaling vod thumbnails into JSON: " + err.Error())
	}

	thumbnailsURL := fmt.Sprintf(thumbnailsURLFmt, p.VinylHost, int(p.vodID))
	createThumbnails := func() error {
		log.Printf("[VOD] Creating thumbs %s -  %s\n", thumbnailsURL, thumbnailsData)
		resp, err := jsonRequest(http.MethodPost, thumbnailsURL, thumbnailsData)
		if err != nil {
			log.Printf("[VOD] Error creating thumbnails for vod %d: %s \n", p.vodID, err)
			reportVinylStatusCode(resp)
			return fmt.Errorf("Error creating thumbnails for vod %d: %s\n", p.vodID, err)
		}
		_ = resp.Body.Close()
		return nil
	}

	exp := backoff.NewExponentialBackOff()
	return backoff.RetryNotify(createThumbnails, exp, func(error, time.Duration) {
		statsd.Inc(statsdThumbnailsAPIError, 1, 1.0)
	})
}

// Update formats with latest fps and bitrate, then return a copy
// (so they can be used in a thread-safe manner)
func (p *Pusher) getFormats(isFinal bool) map[string]format {
	p.formatsMux.Lock()
	defer p.formatsMux.Unlock()

	for label, q := range p.queues {
		format := p.formats[label]
		// Use only derived values if we are finalizing the VOD
		if isFinal {
			format.Bitrate, format.Fps = q.BitrateAndFps()
		} else {
			format.Bitrate = p.qualities[label].Bitrate
			format.Fps = p.qualities[label].Fps
		}

		if label == constTranscodeSource {
			format.MaxIdrInterval = p.PusherSettings.MaxIdrInterval
		}
	}
	fmts := make(map[string]format, len(p.formats))
	for label, f := range p.formats {
		fmts[label] = *f
	}
	return fmts
}

// Return copy of our current thumbnails
// (so they can be used in a thread-safe manner)
func (p *Pusher) getThumbnails() []thumbnail {
	p.thumbnailsMux.Lock()
	thumbs := make([]thumbnail, len(p.thumbnails))
	copy(thumbs, p.thumbnails)
	p.thumbnailsMux.Unlock()
	return thumbs
}

// get VOD Duration (longest duration of any queue)
func (p *Pusher) getDuration() time.Duration {
	var vodDur time.Duration
	for _, q := range p.queues {
		vodDur = max(vodDur, q.Duration())
	}
	return vodDur
}

func (p *Pusher) addStatHelper(stat string, value int64, rate float32) {
	if p.HlsUrlBase != "" {
		statsd.Inc(fmt.Sprintf("%s%s.%s", statsdOriginPrefix, p.ChannelName, stat), value, rate)
	} else {
		statsd.Inc(stat, value, rate)
	}

}

// Send marshalled json. Response body will be closed if there was an error
func jsonRequest(method, url string, data []byte) (*http.Response, error) {
	req, err := http.NewRequest(method, url, bytes.NewReader(data))
	if err != nil {
		return nil, err
	}
	req.Header.Set("Content-Type", "application/json")
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return resp, err
	}
	if err := statusOK(resp); err != nil {
		defer func() { _ = resp.Body.Close() }()
		data, _ := ioutil.ReadAll(resp.Body)
		return resp, errors.Wrap(err, string(data))
	}
	return resp, nil
}

func addPrefix(key, prefix string) string {
	if prefix != "" {
		return fmt.Sprintf("%s_%s", prefix, key)
	}
	hash := sha1.Sum([]byte(key))
	return fmt.Sprintf("%x_%s", hash[:10], key)
}

func max(a, b time.Duration) time.Duration {
	if a > b {
		return a
	}
	return b
}

func (p *Pusher) readThumbnailFromFile(thumb *avdata.Thumbnail) ([]byte, error) {
	data, err := ioutil.ReadFile(thumb.Path)
	if err != nil {
		log.Printf("[VOD] pusher couldn't read thumbnail: %+v\n", thumb)
		statsd.Inc(statsdLostThumb, 1, 1.0)
		return nil, err
	}
	return data, nil
}

func (p *Pusher) readThumbnailFromOrigin(thumb *avdata.Thumbnail) ([]byte, error) {
	client := &http.Client{
		Timeout: p.OriginTimeout,
	}

	resp, err := client.Get(thumb.Path)
	if err != nil {
		log.Printf("[VOD] Error downloading thumbnail from origin: %v", err)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostThumb), fmt.Sprintf("thumbnail read from origin failed: %s", err))
		p.addStatHelper(statsdLostThumb, 1, 1.0)
		return nil, err
	}

	defer func() {
		_ = resp.Body.Close()
	}()

	if resp.StatusCode != http.StatusOK {
		log.Printf("[VOD] Error downloading thumbnail from origin. StatusCode: %d, losing thumbnail", resp.StatusCode)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostThumb), fmt.Sprintf("thumbnail read from origin failed: %s", err))
		p.addStatHelper(statsdLostThumb, 1, 1.0)
		return nil, fmt.Errorf("Origin ReadThumbnail failed with code %d", resp.StatusCode)
	}

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Printf("[VOD] Error reading VOD thumbnail from origin, losing thumbnail: %s", err)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostThumb), fmt.Sprintf("thumbnail read from origin failed: %s", err))
		p.addStatHelper(statsdLostThumb, 1, 1.0)
		return nil, err
	}

	return data, nil
}

// ProcessVodManifest is responsible for writing the vod manifest to S3
func (p *Pusher) ProcessVodManifest(manifestData []byte, manifestPath string) {
	statsd.Inc(statsdNewManifest, 1, 1.0)

	var manifestUploadOperation backoff.Operation
	objName := path.Join(p.VodKey, manifestPath)

	manifestUploadOperation = func() error {
		err := p.Conn.Put(p.Bucket, objName, manifestData, playlistMIMEType)
		if err != nil {
			log.Printf("[VOD][MANIFEST]  couldn't upload manifest %s:%s\n", objName, err)
		}
		return err
	}

	err := backoff.Retry(manifestUploadOperation, backoff.NewExponentialBackOff())
	if err != nil {
		statsd.Inc(statsdLostManifest, 1, 1.0)
	}
}

func reportVinylStatusCode(r *http.Response) {
	if r == nil {
		statsd.Inc(fmt.Sprintf("%s.%s", statsdVinylAPI, statsdVinylDialFailure), 1, 1.0)
	} else if r != nil && statusOK(r) != nil {
		statsd.Inc(fmt.Sprintf("%s.%d", statsdVinylAPI, r.StatusCode), 1, 1.0)
	}
}
