package vod

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"log"
	"math"
	"net/http"
	"path"
	"sync"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/avdata"
	"code.justin.tv/video/gotranscoder/pkg/m3u8"
	"code.justin.tv/video/gotranscoder/pkg/noisy"
	"code.justin.tv/video/gotranscoder/pkg/statsd"
)

const (
	// Content-Type for playlist
	playlistMIMEType = "application/x-mpegURL"

	// Content-Type for *.ts video segments
	videoMIMEType = "video/MP2T"

	//Max duration of lost chunks to be considered healthy
	constMaxLostDuration = 5 * time.Minute
)

const (
	statsdNewChunk           = "vod.new.chunk"
	statsdNewPlaylist        = "vod.new.playlist"
	statsdLostChunk          = "vod.lost.chunk" // Dropped a chunk
	statsdLostPlaylist       = "vod.lost.playlist"
	statsdLostGreedyRead     = "vod.lost.greedy_read"
	statsdLostGreedyWrite    = "vod.lost.greedy_write"
	statsdQueueFull          = "vod.localqueue.full"
	statsdBytesWritten       = "vod.chunks.bytes_written"
	statsdFlushErrors        = "vod.chunks.flush.errors"
	statsdPutTimeFmt         = "vod.put.%s"          // Put timings for a quality
	statsdPlaylistPutTimeFmt = "vod.playlist.put.%s" // Put timings for a playlist quality
	statsdS3ThroughputFmt    = "vod.throughput.%s"   // Throughput for quality

	statsdOriginPrefix = "origin_darklaunch."
)

// chunk represents a collection of segments that will form
// an s3 object and m3u8 playlist entry
type chunk struct {
	segments      []*avdata.Segment
	duration      time.Duration
	sequenceNum   int
	data          bytes.Buffer
	contentLength int
	lost          bool
	frames        int
}

// Add this segment and return new number of segments
func (ch *chunk) Add(seg *avdata.Segment, data []byte, lost bool) int {
	ch.lost = ch.lost || lost

	n, err := ch.data.Write(data)
	if err != nil {
		log.Println(err)
		ch.lost = true
		noisy.Error(statsdLostGreedyWrite, fmt.Sprintf("segment greedy write to bytebuffer failed: %s", err))
	}

	ch.contentLength += n
	ch.frames += int(seg.FrameCount)

	ch.segments = append(ch.segments, seg)
	ch.duration += time.Duration(seg.Duration*1e6) * time.Nanosecond
	return len(ch.segments)
}

func (ch *chunk) Empty() bool {
	return ch == nil || len(ch.segments) == 0
}

// Create a new chunk with incremented sequence number
func (ch *chunk) nextChunk() *chunk {
	return &chunk{
		sequenceNum: ch.sequenceNum + 1,
	}
}

// Queue for processing segments into playlist chunks
// and objects. Stores all subchunk/subobject state
type queue struct {
	*Pusher

	// Queue of chunks to be uploaded
	C chan *chunk

	Notify chan *avdata.Segment

	NotificationWorker sync.WaitGroup

	// Video format (high, medium, ...)
	format string

	// State of sub-chunk
	subChunk *chunk

	// state of current VOD
	playlist *Playlist
	buffer   bytes.Buffer
	stats    struct { // Must Lock() before accessing fields
		sync.Mutex
		Duration time.Duration
		LostDur  time.Duration // Duration lost from marked discontinuities
		MaxChunk time.Duration // Longest chunk seen
		Bytes    int
		Frames   int
	}
}

// newQueue constructs a new queue for a VOD format. Starts the
// queue processing fucntion in a goroutine
func newQueue(p *Pusher, format string) *queue {
	q := &queue{
		Pusher:   p,
		C:        make(chan *chunk, p.QueueSize),
		Notify:   make(chan *avdata.Segment, p.PusherSettings.NotifyQueueSize),
		format:   format,
		subChunk: &chunk{},
		playlist: &Playlist{},
	}
	p.outstandingWorkers.Add(1)
	go q.consume()
	q.NotificationWorker.Add(1)
	go q.handleNotifications()
	return q
}

func (q *queue) handleNotifications() {
	defer q.NotificationWorker.Done()

	for seg := range q.Notify {
		_ = q.AddSegment(seg)
	}
}

// Add segment to our subChunk. If 'seg' belongs to the next chunk,
// flush the old one and create new subchunk
// If HlsUrlBase is specified, download segment from Origin, else
// read from memory
func (q *queue) AddSegment(seg *avdata.Segment) error {
	lost := false
	var data []byte
	var err error

	if q.HlsUrlBase != "" {
		// use Origin
		data, err = q.readSegmentFromOrigin(seg)
		if err != nil {
			lost = true
		}
	} else {
		// use memory
		data, err = q.readSegmentFromMemory(seg)
		if err != nil {
			lost = true
		}
	}

	if q.subChunk.Add(seg, data, lost) < q.ChunkFactor {
		return nil
	}

	if err = q.flushChunk(); err != nil {
		log.Printf("[VOD] error flushing chunk: %v", err)
	}

	q.subChunk = q.subChunk.nextChunk()
	return err
}

func (q *queue) readSegmentFromMemory(seg *avdata.Segment) ([]byte, error) {
	data, err := ioutil.ReadFile(path.Join(q.TranscodePath, seg.Label, seg.SegmentName))
	if err != nil {
		log.Printf("[VOD] Error opening VOD segment, losing chunk: %s", err)
		noisy.Error(statsdLostGreedyRead, fmt.Sprintf("segment greedy read failed: %s", err))
		return nil, err
	}

	return data, nil
}

func (q *queue) readSegmentFromOrigin(seg *avdata.Segment) ([]byte, error) {
	client := &http.Client{
		Timeout: q.OriginTimeout,
	}
	resp, err := client.Get(fmt.Sprintf("%s/%s/%s", q.HlsUrlBase, seg.Label, seg.SegmentName))
	if err != nil {
		log.Printf("[VOD] Error downloading segment from origin, losing chunk: %v", err)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostGreedyRead), fmt.Sprintf("segment greedy read failed: %s", err))
		q.addStatHelper(statsdLostGreedyRead, 1, 1.0)
		return nil, err
	}

	defer func() {
		_ = resp.Body.Close()
	}()

	if resp.StatusCode != http.StatusOK {
		log.Printf("[VOD] Error downloading segment from origin. StatusCode: %d, losing chunk", resp.StatusCode)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostGreedyRead), fmt.Sprintf("segment greedy read failed: %s", err))
		q.addStatHelper(statsdLostGreedyRead, 1, 1.0)
		return nil, fmt.Errorf("Origin ReadSegment failed with code %d", resp.StatusCode)
	}

	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Printf("[VOD] Error reading VOD segment from origin, losing chunk: %s", err)
		noisy.Error(fmt.Sprintf("%s%s", statsdOriginPrefix, statsdLostGreedyRead), fmt.Sprintf("segment greedy read failed: %s", err))
		q.addStatHelper(statsdLostGreedyRead, 1, 1.0)
		return nil, err
	}

	return data, nil
}

// Trigger cleanup and stop accepting new segments.
// Worker goroutine will flush queue then exit.
func (q *queue) Quit() {
	close(q.Notify)
	q.NotificationWorker.Wait()
	if err := q.flushChunk(); err != nil {
		log.Println("[VOD] error flushing last chunks", err)
	}
	close(q.C)
}

// Add 'subChunk' as a chunk in our queue and replace 'subChunk' with empty chunk
func (q *queue) flushChunk() error {
	if q.subChunk.Empty() {
		return nil
	}

	select {
	case q.C <- q.subChunk:
	default:
		lost := <-q.C //pop the oldest element in the queue
		q.appendDiscontinuity()
		q.C <- q.subChunk
		q.addStatHelper(statsdQueueFull, 1, 1.0)
		noisy.Error(statsdQueueFull, "vod queue full")
		return fmt.Errorf("VOD %s queue full. Lost %v of video", q.format, lost.duration)
	}
	return nil
}

// consume incoming segments. This is meant to be run in its own
// goroutine. 'outstandingWorkers' must be incremented before
// launching
func (q *queue) consume() {
	defer q.outstandingWorkers.Done()

	for ch := range q.C {
		q.uploadChunk(ch)
	}
}

// Update our playlist and upload chunk to s3.
func (q *queue) uploadChunk(ch *chunk) {
	// Already seen, ignore
	if ch.sequenceNum < q.playlist.Len() {
		return
	}

	// Add discontinuities for dropped chunks
	// NOTE: appendDiscontinuity() modifies the playlist and changes its size.
	//		It is critical that the new length is checked on every iteration.
	for ch.sequenceNum > q.playlist.Len() {
		q.appendDiscontinuity()
	}

	if ch.lost {
		log.Printf("[VOD] Chunk number %d was marked as lost. Adding discontinuity", ch.sequenceNum)
		q.appendDiscontinuity()
		return
	}

	objName := fmt.Sprintf("%d.ts", q.playlist.Len())
	objKey := q.fmtObjectKey(objName)
	putStatName := fmt.Sprintf(statsdPutTimeFmt, q.format)

	startTime := time.Now()
	err := q.Conn.Put(q.Bucket, objKey, ch.data.Bytes(), videoMIMEType)
	statsd.Timing(putStatName, time.Since(startTime), 1.0)
	if err != nil {
		q.addStatHelper(statsdFlushErrors, 1, 1.0)
		q.appendDiscontinuity()
		log.Printf("[VOD] Error flushing segment [%s] to s3:%s\n", objKey, err)
		noisy.Error(statsdFlushErrors, fmt.Sprintf("Error flushing segment to s3: %s", err))
		return
	}

	//Track throughput. Statsd package uses scales timing values to millisecond
	if int(time.Since(startTime)/time.Millisecond) > 0 {
		putStatName = fmt.Sprintf(statsdS3ThroughputFmt, q.format)
		througput := len(q.buffer.Bytes()) / (int(time.Since(startTime) / time.Millisecond))
		statsd.CustomTiming(putStatName, int64(througput), 1.0)
	}

	// Chunk successfully uploaded. Update stats and playlist
	q.stats.Lock()
	q.stats.Duration += ch.duration
	q.stats.MaxChunk = max(q.stats.MaxChunk, ch.duration)
	q.stats.Bytes += ch.contentLength
	q.stats.Frames += ch.frames
	q.stats.Unlock()

	q.playlist.Append(m3u8.Chunk{
		URL:      objName,
		Duration: ch.duration.Seconds() * 1000,
	})

	q.handlePlaylistUpload() // Don't need to retry since this will be updated later
	q.addStatHelper(statsdNewChunk, 1, 1.0)

	statsd.Gauge(statsdBytesWritten, int64(ch.contentLength), 1.0)
	log.Printf("[VOD] chunk uploaded [Size: %d] [Duration:%d] to %s - %s\n", ch.contentLength, ch.duration, q.VodKey, objKey)
}

// Update our playlist with found discontinuities and upload it to s3.
// This doesn't retry on error
func (q *queue) handlePlaylistUpload() {
	if q.playlist.Len() == 0 {
		return // Don't upload empty playlist
	}

	// Make sure discontinuties from other qualities are reflected in this playlist
	lostDur := q.markDiscontinuities(q.playlist)
	q.stats.Lock()
	q.stats.LostDur += lostDur
	q.stats.Unlock()

	go func() {
		_ = q.uploadPlaylist()
	}()
}

func (q *queue) uploadPlaylist() error {
	putStatName := fmt.Sprintf(statsdPlaylistPutTimeFmt, q.format)
	startTime := time.Now()

	data := q.genPlaylist().GenerateV3()
	name := q.fmtObjectKey(q.PlaylistFileName)

	if putErr := q.Conn.PutNoRetry(q.Bucket, name, data, playlistMIMEType); putErr != nil {
		log.Printf("[VOD] Error flushing playlist to S3: %s\n", putErr)
		q.addStatHelper(statsdLostPlaylist, 1, 1.0)
		return putErr
	}
	statsd.Timing(putStatName, time.Since(startTime), 1.0)
	q.addStatHelper(statsdNewPlaylist, 1, 1.0)
	return nil
}

// Generate serialized m3u8 playlist for this VOD
func (q *queue) genPlaylist() *m3u8.Playlist {
	q.stats.Lock()
	pl := &m3u8.Playlist{
		Chunks:         q.playlist.Copy(),
		StreamDuration: (q.stats.Duration - q.stats.LostDur).Seconds(),
		TargetDuration: int(math.Ceil(q.stats.MaxChunk.Seconds())),
		IsFinal:        true,
		PlaylistType:   m3u8.PlaylistTypeEvent,
	}
	q.stats.Unlock()
	return pl
}

// BitrateAndFps returns the bitrate and fps for this vod in a thread safe way
func (q *queue) BitrateAndFps() (bitrate int, fps float64) {
	q.stats.Lock()
	defer q.stats.Unlock()

	duration := q.stats.Duration.Seconds()
	if duration == 0 {
		return 0, 0
	}
	bitrate = int(float64(8*q.stats.Bytes) / duration)
	fps = float64(q.stats.Frames) / duration
	return bitrate, fps
}

// The duration of this quality is its received duration minus chunks
// lost due to discontinuities encountered in other qualities
func (q *queue) Duration() time.Duration {
	q.stats.Lock()
	duration := q.stats.Duration - q.stats.LostDur
	q.stats.Unlock()
	return duration
}

//Returns true for a healthy upload queue
func (q *queue) Health() bool {
	q.stats.Lock()
	defer q.stats.Unlock()
	if q.stats.LostDur > constMaxLostDuration {
		return false
	}
	if (q.stats.Duration.Minutes() / q.stats.LostDur.Minutes()) < 2.0 {
		return false
	}
	return true
}

// appendDiscontinuity adds a discontinuitiy to the end of the
// playlist and updates our vod-wide set of discontinuities encountered
func (q *queue) appendDiscontinuity() {
	q.discontinuitiesMux.Lock()
	q.discontinuities[q.playlist.Len()] = struct{}{}
	q.discontinuitiesMux.Unlock()

	q.playlist.Append(m3u8.Chunk{
		URL: m3u8.DiscontinuityURL,
	})

	q.addStatHelper(statsdLostChunk, 1, 1.0)
}

// Return the full key of a s3 object from a segment name
func (q *queue) fmtObjectKey(name string) string {
	return path.Join(q.VodKey, q.format, name)
}
