package vod

import (
	"encoding/json"
	"fmt"
	"strings"
	"time"
)

//wrapper required by vinyl api
type registerWrapper struct {
	PastBroadcast registerVODProps `json:"past_broadcast"`
}

// registerVODProperties is the data sent when creating a new VOD
type registerVODProps struct {
	BroadcastID         int               `json:"broadcast_id"`
	BroadcastType       string            `json:"broadcast_type"`
	BroadcasterSoftware string            `json:"broadcaster_software"`
	RecordedOn          time.Time         `json:"recorded_on"`
	Status              string            `json:"status"`
	Channel             string            `json:"channel"`
	OwnerID             uint64            `json:"owner_id"`
	Formats             map[string]format `json:"show_formats"`
	Offset              int               `json:"offset"`
	Duration            time.Duration     `json:"-"`
	URI                 string            `json:"uri"`
	Origin              string            `json:"origin"`
	Manifest            string            `json:"manifest"`
	Deleted             bool              `json:"deleted,omitempty"`
}

// finalizeVODProperties is the data sent when finalizing a completed VOD
type finalizeVODProps struct {
	Thumbnails []thumbnail       `json:"thumbnails"`
	Status     string            `json:"status"`
	Formats    map[string]format `json:"show_formats"`
	Duration   time.Duration     `json:"-"`
	Deleted    bool              `json:"deleted,omitempty"`
}

// Thumbnail is the data sent registering a thumbnail
type thumbnail struct {
	Offset time.Duration `json:"offset"`
	Path   string        `json:"path"`
}

// format represents a VOD quality
type format struct {
	Bitrate            int     `json:"bitrate"`
	Fps                float64 `json:"fps"`
	MaxIdrInterval     int     `json:"max_idr_interval"`
	DisplayName        string  `json:"display_name"`
	PlaylistPreference int     `json:"playlist_preference"`
	// "resolution" serialized from width and height
	Width  int `json:"-"`
	Height int `json:"-"`
	// "codecs" serialized into one field
	VideoCodec string `json:"-"`
	AudioCodec string `json:"-"`
}

func (props *registerVODProps) MarshalJSON() ([]byte, error) {
	type Alias registerVODProps
	return json.Marshal(&struct {
		Duration int `json:"duration"`
		*Alias
	}{
		Duration: int(props.Duration.Seconds()),
		Alias:    (*Alias)(props),
	})
}

func (props *finalizeVODProps) MarshalJSON() ([]byte, error) {
	type Alias finalizeVODProps
	return json.Marshal(&struct {
		Duration int `json:"duration"`
		*Alias
	}{
		Duration: int(props.Duration.Seconds()),
		Alias:    (*Alias)(props),
	})
}

func (t *thumbnail) MarshalJSON() ([]byte, error) {
	type Alias thumbnail
	return json.Marshal(&struct {
		Offset float64 `json:"offset"`
		*Alias
	}{
		Offset: t.Offset.Seconds(),
		Alias:  (*Alias)(t),
	})
}

func (f format) MarshalJSON() ([]byte, error) {
	codecs := make([]string, 0, 2)
	resolution := ""
	if f.VideoCodec != "" {
		codecs = append(codecs, f.VideoCodec)
		resolution = fmt.Sprintf("%dx%d", f.Width, f.Height)
	}
	if f.AudioCodec != "" {
		codecs = append(codecs, f.AudioCodec)
	}

	type Alias format
	return json.Marshal(&struct {
		Resolution string `json:"resolution,omitempty"`
		Codecs     string `json:"codecs"`
		*Alias
	}{
		Resolution: resolution,
		Codecs:     strings.Join(codecs, ","),
		Alias:      (*Alias)(&f),
	})
}
