package vod

import (
	"bytes"
	"fmt"
	"log"
	"time"

	"code.justin.tv/video/gotranscoder/pkg/statsd"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
)

const (
	statsdS3Retry   = "vod.s3.retry" // s3 request failed
	statsdS3RespFmt = "vod.s3.%d"    // format with status code
)

type s3Conn struct {
	exp  *s3.S3 // exponential backoff
	once *s3.S3 // No retries
}

// NewS3Connection creates a new s3 Connection. This connection
// is safe to share among concurrent users.
func NewS3Connection(cfg *aws.Config) (Connection, error) {
	log.Printf("[S3] Initializing connection\n")

	sess, err := session.NewSession(cfg)
	if err != nil {
		log.Println("[VOD][AWSERROR]: Failed to create an aws session")
		return nil, err
	}
	sess.Handlers.Retry.PushBackNamed(request.NamedHandler{
		Name: "statsd",
		Fn: func(r *request.Request) {
			statsd.Inc(statsdS3Retry, 1, 1.0)
			statusCode := fmt.Sprintf(statsdS3RespFmt, r.HTTPResponse.StatusCode)
			statsd.Inc(statusCode, 1, 1.0) //vod.s3.$STATUSCODE
			if awsErr, ok := r.Error.(awserr.Error); ok {
				log.Println("[VOD][AWSERROR]:", awsErr.Code(), awsErr.Message(), awsErr.OrigErr())

				if reqErr, ok := awsErr.(awserr.RequestFailure); ok {
					log.Println("[VOD][AWSERROR][REQUESTID]:", reqErr.RequestID())
					log.Println("[VOD][AWSERROR][HOSTID]:", r.HTTPResponse.Header.Get("X-Amz-Id-2"))
					log.Println("[VOD][AWSERROR][RETRYCOUNT]:", r.RetryCount)
				}
			} else {
				log.Println("[VOD][REQUESTERROR]:", r.Error.Error())
			}
		},
	})

	// include successful requests in statsd
	sess.Handlers.ValidateResponse.PushBackNamed(request.NamedHandler{
		Name: "statsd",
		Fn: func(r *request.Request) {
			statusCode := fmt.Sprintf(statsdS3RespFmt, r.HTTPResponse.StatusCode)
			statsd.Inc(statusCode, 1, 1.0) //vod.s3.$STATUSCODE
		},
	})

	log.Printf("[S3] New S3 connection initialized %+v\n", sess)

	return &s3Conn{
		exp:  s3.New(sess, aws.NewConfig().WithMaxRetries(7)),
		once: s3.New(sess, aws.NewConfig().WithMaxRetries(0)), // Retry attempts
	}, nil
}

func (conn *s3Conn) Put(bucket, objName string, data []byte, contentType string) error {
	//TODO: Restore after finalization
	//_, err := conn.exp.PutObject(genInput(bucket, objName, data, contentType))
	req, resp := conn.exp.PutObjectRequest(genInput(bucket, objName, data, contentType))
	startTime := time.Now()
	err := req.Send()

	//TODO: remove after finalization
	if time.Since(startTime) >= 10*time.Second {
		log.Println("[VOD][SLOWREQUEST][RESPONSE]:", resp.String())
		log.Println("[VOD][SLOWREQUEST][HOSTID]:", req.HTTPResponse.Header.Get("X-Amz-Id-2"))
		log.Println("[VOD][SLOWREQUEST][REQUESTID]:", req.RequestID)
		log.Println("[VOD][SLOWREQUEST][CONTENTLENGTH]:", len(data))
		log.Println("[VOD][SLOWREQUEST][RESPCODE]:", req.HTTPResponse.StatusCode)
		log.Println("[VOD][SLOWREQUEST][RETRYCOUNT]:", req.RetryCount)
		log.Println("[VOD][SLOWREQUEST][REQUESTTIME]:", time.Since(startTime))
	}

	return err
}

func (conn *s3Conn) PutNoRetry(bucket, objName string, data []byte, contentType string) error {
	_, err := conn.once.PutObject(genInput(bucket, objName, data, contentType))
	return err
}

func genInput(bucket, objName string, data []byte, contentType string) *s3.PutObjectInput {
	return &s3.PutObjectInput{
		Bucket:        aws.String(bucket),
		Key:           aws.String(objName),
		ACL:           aws.String(s3.ObjectCannedACLPublicRead),
		Body:          bytes.NewReader(data),
		ContentLength: aws.Int64(int64(len(data))),
		ContentType:   aws.String(contentType),
	}
}
