package logging

import (
	"context"
	"io"
	"io/ioutil"
	"os"

	"github.com/Sirupsen/logrus"
)

// Level specifies the lowest log level which should be output
type Level int

var globalLogger Logger

// The set of log levels available
const (
	DebugLevel Level = iota
	InfoLevel
	WarnLevel
	ErrorLevel
)

func (l Level) String() string {
	switch l {
	case DebugLevel:
		return "debug"
	case InfoLevel:
		return "info"
	case WarnLevel:
		return "warning"
	case ErrorLevel:
		return "error"
	default:
		return "invalid"
	}
}

// Logger presents a logging interface similar to logrus for the origin service
type Logger interface {
	Debug(args ...interface{})
	Debugf(string, ...interface{})

	Info(args ...interface{})
	Infof(string, ...interface{})

	Warn(args ...interface{})
	Warnf(string, ...interface{})

	Error(args ...interface{})
	Errorf(string, ...interface{})

	Fatal(args ...interface{})
	Fatalf(string, ...interface{})

	WithFields(fields Fields) Logger
	WithField(key string, value interface{}) Logger
	WithError(error) Logger

	FieldsFrom(ctx context.Context) Logger

	SetLevel(level Level)
}

// Logger is a custom logger for the origin service, similar to logrus.Logger
type logger struct {
	*logrus.Entry
}

// WithFields sets field values for subsequent log events, preserving any existing fields
func (l *logger) WithFields(fields Fields) Logger {
	return &logger{l.Entry.WithFields(logrus.Fields(fields))}
}

// WithField assigns a single field for this logger
func (l *logger) WithField(key string, value interface{}) Logger {
	return &logger{l.Entry.WithField(key, value)}
}

// WithError is a shortcut to set a log field "error" with a value of err
func (l *logger) WithError(err error) Logger {
	return &logger{l.Entry.WithError(err)}
}

// FieldsFrom is a shortcut for logger.WithFields(logging.FieldsFrom(ctx))
func (l *logger) FieldsFrom(ctx context.Context) Logger {
	return &logger{l.Entry.WithFields(logrus.Fields(FieldsFrom(ctx)))}
}

// SetLevel sets the log level
func (l *logger) SetLevel(level Level) {
	switch level {
	case DebugLevel:
		l.Entry.Logger.SetLevel(logrus.DebugLevel)
	case InfoLevel:
		l.Entry.Logger.SetLevel(logrus.InfoLevel)
	case WarnLevel:
		l.Entry.Logger.SetLevel(logrus.WarnLevel)
	case ErrorLevel:
		l.Entry.Logger.SetLevel(logrus.ErrorLevel)
	}
}

// Fields holds key=value pairs to be included in output log lines
type Fields map[string]interface{}

// SetFields overrides any fields in this field set with the fields passed in
func (f Fields) SetFields(fields Fields) {
	for k, v := range fields {
		f[k] = v
	}
}

type fieldsKey struct{}

// WithFields populates the given context with fields which can be used in
// future logrus calls
func WithFields(ctx context.Context, fields Fields) context.Context {
	newFields := make(Fields)
	for k, v := range FieldsFrom(ctx) {
		newFields[k] = v
	}
	for k, v := range fields {
		newFields[k] = v
	}
	return context.WithValue(ctx, fieldsKey{}, newFields)
}

// FieldsFrom produces any fields which were previously set with WithFields
func FieldsFrom(ctx context.Context) Fields {
	newFields := ctx.Value(fieldsKey{})
	if newFields != nil {
		return newFields.(Fields)
	}
	return make(Fields)
}

// Formatter allows for specific formatters to be configured for use
type Formatter int

// Various formatters which are available
const (
	JSONFormatter Formatter = iota
	TextFormatter
)

// Config various options available for the origin logger
type Config struct {
	// Formatter currently can only be JSONFormatter or TextFormatter and defaults to JSONFormatter
	Formatter Formatter
	// Out will default to os.Stdout if nil
	Out io.Writer
}

// New constructs an origin service logger
func New(cfg Config) Logger {
	l := logrus.New()

	switch cfg.Formatter {
	case JSONFormatter:
		l.Formatter = &logrus.JSONFormatter{}
	case TextFormatter:
		l.Formatter = &logrus.TextFormatter{}
	}

	l.Formatter = &logrus.JSONFormatter{}
	if cfg.Out == nil {
		l.Out = os.Stdout
	} else {
		l.Out = cfg.Out
	}
	return &logger{logrus.NewEntry(l)}
}

// Silent returns a logger which logs to ioutil.Discard for test purposes
func Silent() Logger {
	l := logrus.New()
	l.Out = ioutil.Discard
	return &logger{logrus.NewEntry(l)}
}

func init() {
	globalLogger = New(Config{})
	globalLogger.SetLevel(DebugLevel)
}

// Debug logs the message if debug logging is enabled
func Debug(ctx context.Context, message string, v ...interface{}) {
	logger := globalLogger.FieldsFrom(ctx)
	logger.Debugf(message, v...)
}

// Info logs the messaage if info logging is enabled
func Info(ctx context.Context, message string, v ...interface{}) {
	logger := globalLogger.FieldsFrom(ctx)
	logger.Infof(message, v...)
}

// Warn logs the message if warn logging is enabled
func Warn(ctx context.Context, message string, v ...interface{}) {
	logger := globalLogger.FieldsFrom(ctx)
	logger.Warnf(message, v...)
}
