package logging

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestLogger(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	l.SetLevel(DebugLevel)
	l = l.WithField("single", "field")
	l = l.WithFields(Fields{
		"two":  "fields",
		"work": "too",
	})

	l.Debug("test")
	l.Debugf("test")

	l.Info("test")
	l.Info("test")

	l.Warn("test")
	l.Warnf("test")

	l.Error("test")
	l.Errorf("test")

	fields := map[string]interface{}{
		"single": "field",
		"two":    "fields",
		"work":   "too",
		"msg":    "test",
	}

	levels := []string{"debug", "debug", "info", "info",
		"warning", "warning", "error", "error"}

	dec := json.NewDecoder(buf)
	for _, level := range levels {
		var logLine map[string]interface{}
		err := dec.Decode(&logLine)
		require.NoError(t, err, "all lines should be valid json")

		for k, v := range fields {
			assert.Equal(t, v, logLine[k], "all fields should be present")
		}
		assert.Equal(t, level, logLine["level"], "level should be correct")
		assert.Contains(t, logLine, "time")
	}
	assert.EqualValues(t, 0, buf.Len())
}

func TestLoggerSetLevel(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	for _, lev := range []Level{DebugLevel, InfoLevel, WarnLevel, ErrorLevel} {
		l.SetLevel(lev)
		l.Debug()
		l.Info()
		l.Warn()
		l.Error()
	}

	levels := map[Level]int{
		DebugLevel: 0,
		InfoLevel:  0,
		WarnLevel:  0,
		ErrorLevel: 0,
	}

	dec := json.NewDecoder(buf)
	for {
		var logLine map[string]interface{}
		err := dec.Decode(&logLine)
		if err == io.EOF {
			break
		}
		require.NoError(t, err)

		switch logLine["level"].(string) {
		case "debug":
			levels[DebugLevel]++
		case "info":
			levels[InfoLevel]++
		case "warning":
			levels[WarnLevel]++
		case "error":
			levels[ErrorLevel]++
		}
	}

	require.EqualValues(t, 1, levels[DebugLevel])
	require.EqualValues(t, 2, levels[InfoLevel])
	require.EqualValues(t, 3, levels[WarnLevel])
	require.EqualValues(t, 4, levels[ErrorLevel])
}

func TestFieldContext(t *testing.T) {
	// context with no fields
	ctx := context.Background()
	fields := FieldsFrom(ctx)
	assert.Len(t, fields, 0)
	require.NotNil(t, fields)

	// putting fields in a contex
	fields["key"] = "value"
	ctx = WithFields(ctx, fields)
	fields = FieldsFrom(ctx)
	require.Contains(t, fields, "key")
	require.Equal(t, fields["key"], "value")

	// overwriting fields in a context
	overwrite := make(Fields)
	overwrite["key"] = "notvalue"
	ctx = WithFields(ctx, overwrite)
	fields = FieldsFrom(ctx)
	require.Contains(t, fields, "key")
	require.Equal(t, fields["key"], "notvalue")

	// adding to a fieldset already in a context
	add := make(Fields)
	add["add"] = "value"
	ctx = WithFields(ctx, add)
	fields = FieldsFrom(ctx)
	require.Contains(t, fields, "key")
	require.Contains(t, fields, "add")
	require.Equal(t, fields["key"], "notvalue")
	require.Equal(t, fields["add"], "value")
}

func TestLoggerHelpers(t *testing.T) {
	buf := &bytes.Buffer{}
	l := New(Config{
		Out: buf,
	})

	ctx := WithFields(context.Background(), Fields{
		"key": "value",
	})

	l = l.FieldsFrom(ctx)
	l = l.WithError(fmt.Errorf("errval"))
	l.Info()

	var logLine map[string]interface{}
	err := json.NewDecoder(buf).Decode(&logLine)
	require.NoError(t, err)
	require.Contains(t, logLine, "key")
	require.Contains(t, logLine, "error")
	require.Equal(t, logLine["key"], "value")
	require.Equal(t, logLine["error"], "errval")
}
