package metrics

import (
	"strings"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twirp"
)

var reqStartTimestampKey = new(int)

func markReqStart(ctx context.Context) context.Context {
	return context.WithValue(ctx, reqStartTimestampKey, time.Now())
}

func getReqStart(ctx context.Context) (time.Time, bool) {
	t, ok := ctx.Value(reqStartTimestampKey).(time.Time)
	return t, ok
}

// NewTwirpMWSHooks creates twirp hooks that will report metrics to MWS
func NewTwirpMWSHooks(m *Client) *twirp.ServerHooks {
	hooks := twirp.NewServerHooks()
	// RequestReceived: inc twirp.total.req_recv
	hooks.RequestReceived = func(ctx context.Context) context.Context {
		ctx = markReqStart(ctx)
		_ = m.Record("", "twirp.RequestReceived", "NumCalls", UnitCounter, 1.0)
		return ctx
	}

	// RequestRouted: inc twirp.<method>.req_recv
	hooks.RequestRouted = func(ctx context.Context) context.Context {
		method, ok := twirp.MethodName(ctx)
		if !ok {
			return ctx
		}
		_ = m.Record("", "twirp.RequestRouted.Method."+sanitize(method), "NumCalls", UnitCounter, 1.0)
		return ctx
	}

	hooks.ResponseSent = func(ctx context.Context) context.Context {
		var (
			start  time.Time
			method string
			status string

			haveStart  bool
			haveMethod bool
			haveStatus bool
		)

		start, haveStart = getReqStart(ctx)
		method, haveMethod = twirp.MethodName(ctx)
		status, haveStatus = twirp.StatusCode(ctx)

		method = sanitize(method)
		status = sanitize(status)

		_ = m.Record("", "twirp.ResponseSent", "NumCalls", UnitCounter, 1.0)

		if haveMethod {
			_ = m.Record("", "twirp.ResponseSent.Method."+method, "NumCalls", UnitCounter, 1.0)
		}
		if haveStatus {
			_ = m.Record("", "twirp.ResponseSent.StatusCode."+status, "NumCalls", UnitCounter, 1.0)
		}
		if haveMethod && haveStatus {
			_ = m.Record("", "twirp.ResponseSent.Method."+status+".StatusCode."+status, "NumCalls", UnitCounter, 1.0)
		}

		if haveStart {
			dur := time.Now().Sub(start)
			t := float64(dur) / float64(time.Millisecond)

			_ = m.Record("", "twirp.ResponseSent", "Time", UnitMillisecond, t)

			if haveMethod {
				_ = m.Record("", "twirp.ResponseSent.Method."+method, "Time", UnitMillisecond, t)
			}
			if haveStatus {
				_ = m.Record("", "twirp.ResponseSent.StatusCode."+status, "Time", UnitMillisecond, t)
			}
			if haveMethod && haveStatus {
				_ = m.Record("", "twirp.ResponseSent.Method."+status+".StatusCode."+status, "Time", UnitMillisecond, t)
			}
		}
		return ctx
	}
	return hooks
}

func sanitize(s string) string {
	return strings.Map(sanitizeRune, s)
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}
