package server

import (
	"net/http"
	"path"

	"code.justin.tv/common/twirp"
	"code.justin.tv/common/twirp/hooks/statsd"
	"code.justin.tv/video/lvsapi/internal/auth"
	"code.justin.tv/video/lvsapi/internal/caching"
	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/lvsapi/internal/metrics"
	"code.justin.tv/video/lvsapi/internal/service"

	"code.justin.tv/video/lvsapi/rpc/lvs"
	"code.justin.tv/video/lvsapi/streamkey"

	"goji.io/pat"

	"code.justin.tv/video/lvsapi/internal/awsutils"
	"code.justin.tv/video/lvsapi/internal/digestion"
	"code.justin.tv/video/lvsapi/internal/viewcounts"
	goji "goji.io"
)

// New returns a new lvsapi server. Everything in this sever requires valid auth
func New(secret streamkey.SecretSource, cache caching.CacheInterface, s3APIs awsutils.S3APIs, s statsd.Statter, m *metrics.Client, dgn digestion.DigestionBackend, viewcountsBackend viewcounts.ViewcountsCache) http.Handler {
	mux := goji.NewMux()
	mux.Use(auth.RequireValid)

	hooks := createTwirpHooks(s, m)
	mux.Handle(pat.Post(path.Join(lvs.LiveVideoServicePathPrefix, "*")), lvs.NewLiveVideoServiceServer(service.New(secret, cache, s3APIs, m, dgn, viewcountsBackend), hooks, nil))
	return mux
}

// NewInternal returns a new lvsapi internal server.
func NewInternal(secret streamkey.SecretSource, s statsd.Statter, m *metrics.Client) http.Handler {
	mux := goji.NewMux()

	hooks := createTwirpHooks(s, m)
	mux.Handle(pat.Post(path.Join(lvs.LiveVideoServiceInternalPathPrefix, "*")), lvs.NewLiveVideoServiceInternalServer(service.NewInternal(secret), hooks, nil))
	return mux
}

func createTwirpHooks(s statsd.Statter, m *metrics.Client) *twirp.ServerHooks {
	if s != nil {
		return twirp.ChainHooks(statsd.NewStatsdServerHooks(s), logging.TwirpHooks(), metrics.NewTwirpMWSHooks(m))
	}
	return logging.TwirpHooks()
}
