package service

import (
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/common/twirp"
	"code.justin.tv/video/lvsapi/internal/auth"
	"code.justin.tv/video/lvsapi/internal/logging"
	"code.justin.tv/video/lvsapi/rpc/lvs"
	"code.justin.tv/video/lvsapi/streamkey"
	"github.com/golang/protobuf/ptypes"
)

type serviceInternal struct {
	secretSource streamkey.SecretSource
}

// NewInternal returns an implementation of code.justin.tv.video.lvsapi.rpc.lvs.LiveVideoServiceInternal
func NewInternal(ss streamkey.SecretSource) lvs.LiveVideoServiceInternal {
	return &serviceInternal{secretSource: ss}
}

// AuthorizeBroadcast
func (s *serviceInternal) AuthorizeBroadcast(ctx context.Context, req *lvs.AuthorizeBroadcastRequest) (*lvs.AuthorizeBroadcastResponse, error) {
	logging.Debug(ctx, "Received Request: %+v", req)

	decryptedStreamKey, err := streamkey.Decrypt(ctx, s.secretSource, req.GetStreamkey())
	if err != nil {
		return nil, twirp.NewError(twirp.Unauthenticated, "failed to authorize streamkey")
	}

	// Check if the streamkey is expired
	expiration, err := ptypes.Timestamp(decryptedStreamKey.Priv.GetExpirationTime())
	if err != nil {
		return nil, twirp.InternalError("failed to umarshal expiration time")
	}

	if expiration.Before(time.Now()) || expiration.Equal(time.Now()) {
		return nil, twirp.NewError(twirp.PermissionDenied, "expired stream key")
	}

	resp := &lvs.AuthorizeBroadcastResponse{
		ContentId:               decryptedStreamKey.Priv.GetContentId(),
		S3Bucket:                decryptedStreamKey.Priv.GetS3Bucket(),
		SnsNotificationEndpoint: decryptedStreamKey.Priv.GetSnsNotificationEndpoint(),
		S3Prefix:                decryptedStreamKey.Priv.GetS3Prefix(),
		CustomerId:              decryptedStreamKey.Priv.GetCustomerId(),
		ChannelName:             decryptedStreamKey.Priv.GetChannelName(),
		LowLatency:              decryptedStreamKey.Priv.GetLowLatencyMode(),
		CdnUrl:                  decryptedStreamKey.Priv.GetCdnUrl(),
	}
	return resp, nil
}

func (s *serviceInternal) ResetStreamKeySecret(ctx context.Context, req *lvs.ResetStreamKeySecretRequest) (*lvs.ResetStreamKeySecretResponse, error) {
	logging.Debug(ctx, "Received Request: %+v", req)

	if !auth.IsValidCustomerID(req.GetCustomerId()) {
		return nil, twirp.InvalidArgumentError("customer_id", "customer_id must conform to the LVS API specification")
	}

	newSecret, err := streamkey.GenerateSecret()
	if err != nil {
		return nil, twirp.InternalError("failed to generate new stream secret")
	}

	if err := s.secretSource.Set(ctx, req.GetCustomerId(), newSecret); err != nil {
		return nil, twirp.InternalErrorWith(err)
	}

	return &lvs.ResetStreamKeySecretResponse{}, nil
}
