package service

import (
	"context"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/common/twirp"
	"code.justin.tv/video/lvsapi/rpc/lvs"
	"code.justin.tv/video/lvsapi/streamkey/streamkeyfakes"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestAuthorizeBroadcastValidStreamkey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(svc.internalService())
	defer server.Close()

	client := lvs.NewLiveVideoServiceInternalProtobufClient(server.URL, http.DefaultClient)

	key, err := svc.streamKey(1 * time.Second)
	require.NoError(t, err)

	resp, err := client.AuthorizeBroadcast(context.Background(), &lvs.AuthorizeBroadcastRequest{
		Streamkey: key,
	})

	require.NoError(t, err)
	require.NotNil(t, resp)
}

func TestAuthorizeBroadcastExpiredStreamkey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(svc.internalService())
	defer server.Close()

	client := lvs.NewLiveVideoServiceInternalProtobufClient(server.URL, http.DefaultClient)

	key, err := svc.streamKey(-1 * time.Second)
	require.NoError(t, err)

	_, err = client.AuthorizeBroadcast(context.Background(), &lvs.AuthorizeBroadcastRequest{
		Streamkey: key,
	})

	require.EqualValues(t, err, twirp.NewError(twirp.PermissionDenied, "expired stream key"))
}

func TestAuthorizeBroadcastNilStreamkey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(lvs.NewLiveVideoServiceInternalServer(NewInternal(svc.secretSource), nil, nil))
	defer server.Close()

	client := lvs.NewLiveVideoServiceInternalProtobufClient(server.URL, http.DefaultClient)

	_, err = client.AuthorizeBroadcast(context.Background(), nil)
	require.Error(t, err)
}

func TestAuthorizeBroadcastInvalidStreamkey(t *testing.T) {
	svc, err := mockService()
	require.NoError(t, err)

	server := httptest.NewServer(lvs.NewLiveVideoServiceInternalServer(NewInternal(svc.secretSource), nil, nil))
	defer server.Close()

	client := lvs.NewLiveVideoServiceInternalProtobufClient(server.URL, http.DefaultClient)
	_, err = client.AuthorizeBroadcast(context.Background(), &lvs.AuthorizeBroadcastRequest{
		Streamkey: "test-string",
	})

	require.EqualValues(t, err, twirp.NewError(twirp.Unauthenticated, "failed to authorize streamkey"))
}

func TestResetStreamKeySecret(t *testing.T) {
	secretSource := &streamkeyfakes.FakeSecretSource{}
	server := httptest.NewServer(lvs.NewLiveVideoServiceInternalServer(NewInternal(secretSource), nil, nil))
	defer server.Close()

	// the First set call returns a nil error (mimicking success)
	secretSource.SetReturnsOnCall(0, nil)

	client := lvs.NewLiveVideoServiceInternalProtobufClient(server.URL, http.DefaultClient)
	_, err := client.ResetStreamKeySecret(context.Background(), &lvs.ResetStreamKeySecretRequest{
		CustomerId: "test-customer",
	})
	require.NoError(t, err)

	// make sure set was actually called
	require.Equal(t, secretSource.SetCallCount(), 1)
	_, customerID, _ := secretSource.SetArgsForCall(0)
	assert.Equal(t, customerID, "test-customer")

}
