package usher

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/video/lvsapi/internal/httputil"
)

// Usher is just a struct which ties up the API's and make testing easier
type Usher struct {
	client  httputil.HttpClient
	statter statsd.Statter
}

// NewUsher function returns instance of Usher with the client set
func NewUsher(s statsd.Statter) *Usher {
	usher := Usher{
		client:  &http.Client{},
		statter: s,
	}
	return &usher
}

// UsherStreamResponse holds the basic usher data to populate an LVS response
type UsherStreamResponse struct {
	Channel      string `json:"channel"`
	StartedOn    int64  `json:"started_on"`
	ChannelCount int    `json:"channel_count"`
	LVSMetadata  string `json:"lvs_metadata"`
	Status       string `json:"status"`
	CustomerId   string `json:"customer_id"`
	ContentId    string `json:"content_id"`
	StreamId     uint64 `json:"destination"`
}

// ListStreams will return a list of usher streams for given customerId
func (u *Usher) ListStreams(customerId string) ([]UsherStreamResponse, error) {
	url := fmt.Sprintf("%s/%s.json", "http://usher.justin.tv/hls_transcode/lvs_customer_transcodes", customerId)

	// Track latency
	start := time.Now()
	rsp, err := u.client.Get(url)
	if u.statter != nil {
		_ = u.statter.TimingDuration("usher.list_streams", time.Since(start), 1.0)
	}

	if err != nil {
		return nil, err
	}
	defer func() {
		if err := rsp.Body.Close(); err != nil {
			fmt.Printf("Unable to close response body %s", err)
		}
	}()

	if rsp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("returned error code : %s ", rsp.StatusCode)
	}

	list, err := ioutil.ReadAll(rsp.Body)
	if err != nil {
		return nil, err
	}

	var lst []UsherStreamResponse

	// don't try to unmarshal usher's non compliant empty response
	if len(list) < 3 {
		return lst, nil
	}

	if err := json.Unmarshal(list, &lst); err != nil {
		return nil, err
	}

	return lst, nil
}

// GetStreamInfo return the information around a given usher stream in its streams table
func (u *Usher) GetStreamInfo(customerId string, contentId string) (*UsherStreamResponse, error) {
	url := fmt.Sprintf("%s/%s/%s.json", "http://usher.justin.tv/hls_transcode/lvs_channel", customerId, contentId)
	// Track latency
	start := time.Now()
	rsp, err := u.client.Get(url)
	if u.statter != nil {
		_ = u.statter.TimingDuration("usher.get_stream_info", time.Since(start), 1.0)
	}

	if err != nil {
		return nil, err
	}
	defer func() {
		if err := rsp.Body.Close(); err != nil {
			fmt.Printf("Unable to close response body %s", err)
		}
	}()

	if rsp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("returned error code : %s ", rsp.StatusCode)
	}

	st, err := ioutil.ReadAll(rsp.Body)
	if err != nil {
		return nil, err
	}

	var str []UsherStreamResponse

	// don't try to unmarshal usher's non compliant empty response
	if len(st) < 3 {
		return nil, errors.New("Stream Not found")
	}

	if err := json.Unmarshal(st, &str); err != nil {
		return nil, err
	}

	return &str[0], nil
}
