package streamkey

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"path"
	"testing"
	"time"

	"code.justin.tv/jorge/clock"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type mockS3Client struct {
	s3iface.S3API

	getCount int
	putCount int

	objects map[string]map[string][]byte
}

func (m *mockS3Client) GetObjectWithContext(ctx aws.Context, input *s3.GetObjectInput, opts ...request.Option) (*s3.GetObjectOutput, error) {
	m.getCount++

	bucket, ok := m.objects[aws.StringValue(input.Bucket)]
	if !ok {
		return nil, fmt.Errorf("invalid bucket %s", aws.StringValue(input.Bucket))
	}

	data, ok := bucket[aws.StringValue(input.Key)]
	if !ok {
		return nil, fmt.Errorf("invalid key %s", aws.StringValue(input.Key))
	}

	return &s3.GetObjectOutput{
		Body: ioutil.NopCloser(bytes.NewReader(data)),
	}, nil
}

func (m *mockS3Client) PutObjectWithContext(ctx aws.Context, input *s3.PutObjectInput, opts ...request.Option) (*s3.PutObjectOutput, error) {
	m.putCount++

	if m.objects == nil {
		m.objects = make(map[string]map[string][]byte)
	}

	if _, ok := m.objects[aws.StringValue(input.Bucket)]; !ok {
		m.objects[aws.StringValue(input.Bucket)] = make(map[string][]byte)
	}

	data, err := ioutil.ReadAll(input.Body)
	if err != nil {
		return nil, err
	}

	m.objects[aws.StringValue(input.Bucket)][aws.StringValue(input.Key)] = data

	return &s3.PutObjectOutput{}, nil
}

func TestGet(t *testing.T) {
	customerID := "test-cust"
	s3Prefix := "some/prefix"
	s3Bucket := "test-bucket"
	s3Key := path.Join(s3Prefix, customerID+".pem")

	secret, err := GenerateSecret()
	require.NoError(t, err)

	buf := &bytes.Buffer{}
	require.NoError(t, secret.Encode(buf))

	var mocktime *clock.Mock
	var svc *mockS3Client
	var ss *S3SecretSource

	mocktime = clock.NewMock()

	svc = &mockS3Client{
		objects: map[string]map[string][]byte{
			s3Bucket: map[string][]byte{
				s3Key: buf.Bytes(),
			},
		},
	}

	ss = NewS3SecretSource(S3SecretSourceConfig{
		S3:           svc,
		Clock:        mocktime,
		Bucket:       s3Bucket,
		Prefix:       s3Prefix,
		CacheTimeout: 10 * time.Second,
	})

	// trigger an s3 fetch first
	s, err := ss.Get(context.Background(), customerID)
	require.NoError(t, err)
	assert.Equal(t, secret.bytes, s.bytes)

	// now fetch again, this should read from the cache
	s, err = ss.Get(context.Background(), customerID)
	require.NoError(t, err)
	assert.Equal(t, secret.bytes, s.bytes)

	// make sure we only made one get request to this point
	assert.Equal(t, svc.getCount, 1)

	// timeout is 10 secs, this ensures the content is expired
	mocktime.Add(20 * time.Second)

	// trigger a fresh fetch
	s, err = ss.Get(context.Background(), customerID)
	require.NoError(t, err)
	assert.Equal(t, secret.bytes, s.bytes)

	// we should now have seen two s3 gets
	assert.Equal(t, svc.getCount, 2)
}

func TestGetS3Error(t *testing.T) {
	customerID := "test-cust"
	s3Prefix := "some/prefix"
	s3Bucket := "test-bucket"

	var mocktime *clock.Mock
	var svc *mockS3Client
	var ss *S3SecretSource

	mocktime = clock.NewMock()
	svc = &mockS3Client{}
	ss = NewS3SecretSource(S3SecretSourceConfig{
		S3:           svc,
		Clock:        mocktime,
		Bucket:       s3Bucket,
		Prefix:       s3Prefix,
		CacheTimeout: 10 * time.Second,
	})

	// trigger an s3 fetch first
	_, err := ss.Get(context.Background(), customerID)
	assert.Error(t, err)
}

func TestSet(t *testing.T) {
	customerID := "test-cust"
	s3Prefix := "some/prefix"
	s3Bucket := "test-bucket"
	s3Key := path.Join(s3Prefix, customerID+".pem")

	var mocktime *clock.Mock
	var svc *mockS3Client
	var ss *S3SecretSource

	mocktime = clock.NewMock()
	svc = &mockS3Client{}
	ss = NewS3SecretSource(S3SecretSourceConfig{
		S3:           svc,
		Clock:        mocktime,
		Bucket:       s3Bucket,
		Prefix:       s3Prefix,
		CacheTimeout: 10 * time.Second,
	})

	secret, err := GenerateSecret()
	require.NoError(t, err)

	// trigger an s3 fetch first
	err = ss.Set(context.Background(), customerID, secret)
	assert.NoError(t, err)

	// ensure we wrote the secret
	s := new(Secret)
	err = s.Decode(bytes.NewReader(svc.objects[s3Bucket][s3Key]))
	require.NoError(t, err)

	// ensure the written secret is valid
	assert.Equal(t, secret.bytes, s.bytes)

	// trigger a secret get, which should be cached
	s, err = ss.Get(context.Background(), customerID)
	require.NoError(t, err)
	assert.Equal(t, secret.bytes, s.bytes)
	assert.Equal(t, svc.getCount, 0)
}
