package streamkey

import (
	"context"
	"fmt"
)

// SecretSource is the interface used to hold customer specific stream key secrets
type SecretSource interface {
	Get(ctx context.Context, customerID string) (*Secret, error)
	Set(ctx context.Context, customerID string, secret *Secret) error
}

// MapSecretSource provides a simple interface a static map of secrets
type MapSecretSource map[string]*Secret

// Get performs a simple map get to fetch secrets
func (m MapSecretSource) Get(ctx context.Context, customerID string) (*Secret, error) {
	s, ok := m[customerID]
	if !ok {
		return nil, fmt.Errorf("no such secret")
	}

	return s, nil
}

// Set performs a simple map set to set secrets
func (m MapSecretSource) Set(ctx context.Context, customerID string, secret *Secret) error {
	m[customerID] = secret
	return nil
}

// SingleSecretSource provides an interface that causes Get to always return the same secret
type SingleSecretSource struct {
	*Secret
}

// Get always returns the same secret
func (ss *SingleSecretSource) Get(ctx context.Context, customerID string) (*Secret, error) {
	return ss.Secret, nil
}

// Set always returns an error as it is unsupported for SingleSecretSource
func (ss *SingleSecretSource) Set(ctx context.Context, customerID string, secret *Secret) error {
	return fmt.Errorf("invalid operation for SingleSecretSource")
}
