package streamkey

import (
	"context"
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"encoding/base64"
	"fmt"
	"io"
	"strings"

	"github.com/golang/protobuf/proto"
	"github.com/pkg/errors"
)

var (
	base64encoding = base64.RawURLEncoding
)

// StreamKey holds an un
type StreamKey struct {
	Pub  *PublicData
	Priv *PrivateData
}

// NewV1 constructs a new V1 lvs stream key
func NewV1(customerID string, priv *PrivateData) *StreamKey {
	return &StreamKey{
		Pub: &PublicData{
			Version:    Version_V1,
			CustomerId: customerID,
		},
		Priv: priv,
	}
}

// Encrypt encrypts a stream key configuration and produces a valid stream key
func Encrypt(ctx context.Context, ss SecretSource, key *StreamKey) (string, error) {
	pubMsg, err := proto.Marshal(key.Pub)
	if err != nil {
		return "", errors.Wrap(err, "failed to marshal pub data")
	}

	privMsg, err := proto.Marshal(key.Priv)
	if err != nil {
		return "", errors.Wrap(err, "failed to marshal priv data")
	}

	secret, err := ss.Get(ctx, key.Pub.GetCustomerId())
	if err != nil {
		return "", errors.Wrap(err, "failed to fetch streamkey secret")
	}

	aesCipher, err := aes.NewCipher(secret.bytes[:])
	if err != nil {
		return "", errors.Wrap(err, "failed to init cipher")
	}

	gcm, err := cipher.NewGCM(aesCipher)
	if err != nil {
		return "", errors.Wrap(err, "failed to init cipher")
	}

	nonce := make([]byte, gcm.NonceSize())
	if _, err = io.ReadFull(rand.Reader, nonce); err != nil {
		return "", errors.Wrap(err, "failed to generate nonce from random")
	}

	// seal and prepend the nonce
	privMsg = gcm.Seal(privMsg[:0], nonce, privMsg, pubMsg)
	privMsg = append(nonce, privMsg...)

	pubMsgStr := base64encoding.EncodeToString(pubMsg)
	privMsgStr := base64encoding.EncodeToString(privMsg)

	return fmt.Sprintf("lvs.%s.%s", pubMsgStr, privMsgStr), nil
}

// Decrypt parses a stream key and returns a decrypted stream key configuration
func Decrypt(ctx context.Context, ss SecretSource, key string) (*StreamKey, error) {
	keyParts := strings.Split(key, ".")
	if keyParts[0] != "lvs" || len(keyParts) != 3 {
		return nil, fmt.Errorf("invalid stream key format")
	}

	pubMsgStr := keyParts[1]
	privMsgStr := keyParts[2]

	pubMsg, err := base64encoding.DecodeString(pubMsgStr)
	if err != nil {
		return nil, errors.Wrap(err, "malformed stream key")
	}

	privMsg, err := base64encoding.DecodeString(privMsgStr)
	if err != nil {
		return nil, errors.Wrap(err, "malformed stream key")
	}

	pub := &PublicData{}

	if err = proto.Unmarshal(pubMsg, pub); err != nil {
		return nil, errors.Wrap(err, "failed to unmarshal pub data")
	}

	secret, err := ss.Get(ctx, pub.GetCustomerId())
	if err != nil {
		return nil, errors.Wrap(err, "failed to fetch streamkey secret")
	}

	aesBlock, err := aes.NewCipher(secret.bytes[:])
	if err != nil {
		return nil, errors.Wrap(err, "failed to init cipher")
	}

	gcm, err := cipher.NewGCM(aesBlock)
	if err != nil {
		return nil, errors.Wrap(err, "failed to init cipher")
	}

	if len(privMsg) < gcm.NonceSize() {
		return nil, errors.New("malformed stream key")
	}

	// split out the nonce
	nonce := privMsg[:gcm.NonceSize()]
	privMsg = privMsg[gcm.NonceSize():]

	privMsg, err = gcm.Open(privMsg[:0], nonce, privMsg, pubMsg)
	if err != nil {
		return nil, errors.Wrap(err, "invalid stream key")
	}

	priv := &PrivateData{}

	if err := proto.Unmarshal(privMsg, priv); err != nil {
		return nil, errors.Wrap(err, "failed to unmarshal priv data")
	}

	return &StreamKey{Pub: pub, Priv: priv}, nil
}

// GetCustomerIdFromKey just extracts the customer id from the streamkey
func GetCustomerIdFromKey(key string) (string, error) {
	keyParts := strings.Split(key, ".")
	if keyParts[0] != "lvs" || len(keyParts) != 3 {
		return "", fmt.Errorf("invalid stream key format")
	}

	pubMsgStr := keyParts[1]

	pubMsg, err := base64encoding.DecodeString(pubMsgStr)
	if err != nil {
		return "", errors.Wrap(err, "malformed stream key")
	}

	pub := &PublicData{}

	if err = proto.Unmarshal(pubMsg, pub); err != nil {
		return "", errors.Wrap(err, "failed to unmarshal pub data")
	}

	return pub.CustomerId, nil
}
