package httpserver

import (
	"encoding/json"
	"net/http"

	"github.com/dimfeld/httptreemux"
)

type HTTPServer struct {
	Router     *httptreemux.ContextMux
	Middleware []Middleware
}

type HTTPError struct {
	Err     string `json:"Error"`
	Code    int    `json:"Code"`
	Message string `json:"Message"`
	Cause   string `json:"-"`
}

func (h HTTPError) Error() string {
	return h.Err
}

type Middleware func(h http.Handler) http.Handler

type HandlerFunc func(*http.Request, map[string]string) (interface{}, *HTTPError)

func New() *HTTPServer {
	server := &HTTPServer{
		Router:     httptreemux.NewContextMux(),
		Middleware: DefaultMiddleware(),
	}
	registerPingHandler(server.Router)
	ConfigureLogging()
	return server
}

func DefaultMiddleware() []Middleware {
	return []Middleware{
		RequestID,
		CORS,
		Recoverer,
		RequestLog,
		DefaultRequestSizeLimiter,
	}
}

func ApplyMiddleware(router http.Handler, middleware []Middleware) http.Handler {
	handler := router
	for _, f := range middleware {
		handler = f(handler)
	}
	return handler
}

func (s *HTTPServer) Start() error {
	handler := ApplyMiddleware(s.Router, s.Middleware)
	logger.Info("Started HTTP server on port 8000...")
	return http.ListenAndServe(":8000", handler)
}

func (s *HTTPServer) GET(path string, handler HandlerFunc) {
	s.Router.GET(path, makeJSONHandler(handler))
}

func (s *HTTPServer) POST(path string, handler HandlerFunc) {
	s.Router.POST(path, makeJSONHandler(handler))
}

func (s *HTTPServer) PUT(path string, handler HandlerFunc) {
	s.Router.PUT(path, makeJSONHandler(handler))
}

func (s *HTTPServer) DELETE(path string, handler HandlerFunc) {
	s.Router.DELETE(path, makeJSONHandler(handler))
}

func (s *HTTPServer) PATCH(path string, handler HandlerFunc) {
	s.Router.PATCH(path, makeJSONHandler(handler))
}

func makeJSONHandler(handler HandlerFunc) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		out, httperr := handler(r, httptreemux.ContextParams(r.Context()))
		respondJSON(w, out, httperr)
	}
}

func respondJSON(w http.ResponseWriter, obj interface{}, httperr *HTTPError) {
	w.Header().Set("Content-Type", "application/json")
	if httperr != nil {
		w.WriteHeader(httperr.Code)
		obj = httperr
		if httperr.Code == http.StatusInternalServerError {
			logger.Error("Error processing request: ", httperr.Cause)
		}
	}

	if obj == nil {
		return
	}

	jsonString, err := json.Marshal(obj)
	if err != nil {
		code := http.StatusInternalServerError
		http.Error(w, http.StatusText(code), code)
		return
	}
	w.Write(jsonString)
}

func registerPingHandler(router *httptreemux.ContextMux) {
	router.GET("/ping", func(w http.ResponseWriter, r *http.Request) {
		w.Write([]byte("pong"))
	})
}
