package probe

import (
	"encoding/json"
	"errors"
	"fmt"
	"time"
)

var (
	errNoVideoStreams = errors.New("no video streams detected")
)

func parseFFprobeError(output []byte) (*ffprobeErr, error) {
	type errWrapper struct {
		Error *ffprobeErr
	}

	raw := new(errWrapper)
	if err := json.Unmarshal(output, raw); err != nil {
		return nil, err
	}
	return raw.Error, nil
}

func parseFFprobeOutput(output []byte) (VideoMetadata, error) {
	raw := new(ffprobeOutput)
	m := VideoMetadata{}
	if err := json.Unmarshal(output, raw); err != nil {
		return m, err
	}

	for _, s := range raw.Streams {
		if s.CodecType == "video" {
			m.VideoStreams = append(m.VideoStreams, Video{
				Codec:    CodecName(s.CodecName),
				Width:    s.width(),
				Height:   s.height(),
				BitRate:  s.bitrate(),
				Duration: s.duration(),
			})
		}
		if s.CodecType == "audio" {
			m.AudioStreams = append(m.AudioStreams, Audio{
				Codec:    CodecName(s.CodecName),
				BitRate:  s.bitrate(),
				Duration: s.duration(),
			})
		}
	}
	if len(m.VideoStreams) == 0 {
		return m, errNoVideoStreams
	}
	return m, nil
}

type ffprobeOutput struct {
	Streams []*stream
}

type stream struct {
	CodecName string      `json:"codec_name"`
	CodecType string      `json:"codec_type"`
	Width     json.Number `json:"width"`
	Height    json.Number `json:"height"`
	BitRate   json.Number `json:"bit_rate"`
	Duration  json.Number `json:"duration"`
}

func (s *stream) width() int {
	w, err := s.Width.Int64()
	if err != nil {
		return 0
	}
	return int(w)
}

func (s *stream) height() int {
	h, err := s.Height.Int64()
	if err != nil {
		return 0
	}
	return int(h)
}

func (s *stream) bitrate() int {
	b, err := s.BitRate.Int64()
	if err != nil {
		return 0
	}
	return int(b)
}

func (s *stream) duration() time.Duration {
	d, err := s.Duration.Float64()
	if err != nil {
		return 0
	}
	return time.Duration(d * float64(time.Second))
}

type ffprobeErr struct {
	Code   int
	String string
}

func (err *ffprobeErr) Error() string {
	return fmt.Sprintf("ffprobe error code=%d: %s", err.Code, err.String)
}
